#include "mbed.h"
#include "rtos.h"
#include "QEI.h"
#include "PID.h"
#include "math.h"

#define RATE 0.1

//Photointerrupter input pins
#define I1pin D2
#define I2pin D11
#define I3pin D12

//Incremental encoder input pins
#define CHA   D7
#define CHB   D8
#define CHA1  D7
#define CHB1  D8

//Motor Drive output pins   //Mask in output byte
#define L1Lpin D4           //0x01
#define L1Hpin D5           //0x02
#define L2Lpin D3           //0x04
#define L2Hpin D6           //0x08
#define L3Lpin D9           //0x10
#define L3Hpin D10          //0x20

#define LED D13

//Mapping from sequential drive states to motor phase outputs
/*
State   L1  L2  L3
0       H   -   L
1       -   H   L
2       L   H   -
3       L   -   H
4       -   L   H
5       H   L   -
6       -   -   -
7       -   -   -
*/

/////////////////////////////////////////
/////////   User Settings   //////////////
/////////////////////////////////////////


volatile float TargetRotation_user = 9999.0;
volatile float TargetSpeed_user = 5.0;



/////////////////////////////////////////
//////////  User Settings   /////////////
/////////////////////////////////////////


PID controller(16.0, 0.5, 0.00625, RATE);
PID slow_controller(3.5, 0.5, 0.00625, RATE);

QEI wheel(CHA1,CHB1,NC,468, QEI::X4_ENCODING);

//Drive state to output table
const int8_t driveTable[] = {0x12,0x18,0x09,0x21,0x24,0x06,0x00,0x00};

//Mapping from interrupter inputs to sequential rotor states. 0x00 and 0x07 are not valid
const int8_t stateMap[] = {0x07,0x05,0x03,0x04,0x01,0x00,0x02,0x07};
//const int8_t stateMap[] = {0x07,0x01,0x03,0x02,0x05,0x00,0x04,0x07}; //Alternative if phase order of input or drive is reversed

//Phase lead to make motor spin
int8_t lead = -2;  //2 for forwards, -2 for backwards

const float notes[15] = {
    3322.44,3520.00,3729.31,3951.07,
    4186.01,4186.01,4434.92,4698.64,
    4978.03,5274.04,5587.65,5587.65,
    5919.91,6271.93,6644.88
};

//Status LED
DigitalOut led1(LED1);
DigitalOut TOG(LED);

//Photointerrupter inputs
DigitalIn I1(I1pin);
DigitalIn I2(I2pin);
DigitalIn I3(I3pin);

//Motor Drive outputs
DigitalOut L1LD(L1Lpin);
DigitalOut L1HD(L1Hpin);
DigitalOut L2LD(L2Lpin);
DigitalOut L2HD(L2Hpin);
DigitalOut L3LD(L3Lpin);
DigitalOut L3HD(L3Hpin);

//PWM Motor Drive outputs
PwmOut L1L(L1Lpin);
PwmOut L1H(L1Hpin);
PwmOut L2L(L2Lpin);
PwmOut L2H(L2Hpin);
PwmOut L3L(L3Lpin);
PwmOut L3H(L3Hpin);

//
volatile float rps = 0.0;
volatile float outputrps = 0.0;
volatile int8_t curState;
volatile int cur_quadrature_state;

volatile float MEDOT = 0.0;
//Declare threads
Thread thread_melody;



//Global counter
volatile int count = 0;
volatile int rotation_number = 0;
volatile int prec_rotation_number = 0;
volatile int total_prec_rotation = 0;

volatile double photo_timer = 0.0;
volatile double photo_rotation_time = 0.0;
volatile double old_photo_timer = 0.0;
volatile double high_res_RPS = 0.0;
volatile float pwm_period = float(1.0/200.0);



//Declare timers
Timer photorpsTimer;
Timer globalTimer;
Timer medoTimer;
Timer mainTimer;



//Declare init
//Initialise the serial port
Serial pc(SERIAL_TX, SERIAL_RX);

InterruptIn photoInt1(I1pin);
InterruptIn photoInt2(I2pin);
InterruptIn photoInt3(I3pin);

//Encoder Interrupt Input
InterruptIn CHA_I(CHA);
InterruptIn CHB_I(CHB);

volatile int8_t orState = 0;    //Rotot offset at motor state 0

volatile int n;
int pidcount;
int photorps = 0;
//Initialise the serial port

volatile int8_t intState = 0;
volatile float outpwm = 0.0;
volatile float slow_down_speed = 0.0;
volatile double position_degree = 0.0;
volatile double temp_timer = 0.0;
volatile double one_rotation_time = 0.0;
volatile double old_p_timer = 0.0;

//Global for melody
volatile float a[8] = {0.0};
volatile int b[8] = {0};
volatile int melody_sum = 0;
volatile int melody_play = 0;
volatile float frequency = 200.0;

//Global for read_input_command
volatile float rev = TargetRotation_user;
volatile float vel = TargetSpeed_user;   // value for revolution and velocity
volatile int r = -1;                     // value for rotate direction

//Global for controller_thread
volatile int Current_rotation = 0;
volatile float Current_res = 0.0;
volatile float dis_to_target = 0.0;

void read_input_command ()
{
    /* Receive Input Command First */
    char n[64] = "";

    // Max's initialisation
    int i = 0,irev = 0,frev = 0,ivel = 0,fvel = 0;
    // r = rotate direction
    // irev = integer part of rotation
    // frev = fractional part of rotation
    // irev = integer part of velocity
    // fvel = fractional part of velocity

    //float rev=0.0,vel = 0.0;
    //float a[8] = {0.0};
    //int b[8] = {0};


    // If there is input then start the parsing system
    if (pc.readable()) {

        // Collect input here
        while (pc.readable()) {
            pc.scanf("%s",&n);
        }
        pc.printf("Input Received: %s \n\r", n);

        // Start parsing here
        if(n[0] != 'T') {       // Check motor or melody command
            if(n[0] == 'R') {   // Check rotation case
                //-
                i++;
                if(n[1]=='-') { // Check forward or backward spinning
                    r = 1;     // Set rotation clockwise
                    i++;
                } else r = -1;   // Set rotation anti-clockwise

                // Check number of roation received (integer part)
                while(n[i]>47&&n[i]<58&&i<4.5+0.5*r) {
                    irev = 10 * irev + (n[i]-48);
                    i++;
                }


                //.
                if(n[i]=='.') { // Check is there decimal in rotation command
                    //digits
                    i++;
                    if(n[i]>47&&n[i]<58) {  // first digit
                        frev = 10*frev + (n[i]-48);
                        i++;
                    }
                    if(n[i]>47&&n[i]<58) {  // second digit
                        frev = 10*frev + (n[i]-48);
                        i++;
                    }
                }
                // recalculate rotation value with fractional part
                rev = float(float(irev) + float(float(frev)/(frev>9?100.0:10.0)));
                rev = -rev * r;  // set direction

                rotation_number = 0;
                pc.printf("R: %3.2f \n\r", rev); // debug
            }
            if(n[i] == 'V') {   // start velocity case here
                //digits
                int j = 1;
                while(n[i+j]>47&&n[i+j]<58&&j<4) {
                    ivel = 10 * ivel+ (n[i+j]-48);  // set integer part
                    j++;
                }
                i+=j;
                //.
                if(n[i]=='.') {   // check whether fractional part exist
                    //digits
                    i++;
                    if(n[i]>47&&n[i]<58) {
                        fvel = 10*fvel + (n[i]-48);    // first digit
                        i++;
                    }
                    if(n[i]>47&&n[i]<58) {
                        fvel = 10*fvel + (n[i]-48);    // second digit
                        i++;
                    }
                }
                // recalculate velocity
                vel = float(float(ivel) + float(float(fvel)/(frev>9?100.0:10.0)));
                pc.printf("V: %2.2f \n\r", vel);  // debug
            }

        } else if (n[0] == 'T') {   // start melody case here
            i=0;
            int j = 1;
            while(n[2*i+j]>64&&n[2*i+j]<72&&i<16) {  // set condition (A-G, #, ^)
                if(n[2*i+j+1]=='#'||n[2*i+j+1]=='^') { // # or ^ case
                    //a[i] = float(int(n[2*i+j])) - 64.0 + (n[2*i+j+1]=='#'?0.5:-0.5) + 7*(n[2*i+j+2]-49); // offset 0.5
                    a[i]= float(int(n[2*i+j])) - 64.0 + (n[2*i+j+1]=='#'?0.5:-0.5); // store character
                    b[i]= n[2*i+j+2]-48; // store duration
                    j++;
                    if(n[2*i+j+2]>47&&n[2*i+j+2]<58){
                        b[i] = b[i]*10 + n[2*i+j+2];
                        j++;
                    }
                } else {
                    a[i] = float(int(n[2*i+j])) - 64.0 ; // store characterTB
                    b[i]= n[2*i+j+2]-48; // store duration
                    j++;
                    if(n[2*i+j+2]>47&&n[2*i+j+2]<58){
                        b[i] = b[i]*10 + n[2*i+j+2];
                        j++;
                    }
                }
                i++;
            }
            if (i != 0) {
                melody_sum = i;
                melody_play = 1;
            }
            //  for(int k=0; k<i; k++) {
            //      pc.printf("Note: %.1f Duration: %d Notes: %d \n\r", a[k], b[k], melody_sum);  // debug
            //  }
        }
        if ( (n[0] != 'R') &&  (n[0] != 'V') && (n[0] != 'T')) {
            pc.printf("Command is not recognised. \n\r");
        }
    }   // if readable end here

    /* Input Command System ends here */
}
void play_melody()
{
    while(1) {

        if(melody_play == 1) {
        medoTimer.reset();
        medoTimer.start();
            if(melody_sum !=0) {
                for(int i=0; i< melody_sum; i++) {
                    frequency = notes[int(a[i]*2)-1];

                    // set duration here
                    pc.printf(" Play %.2f Hz for %d \n\r", frequency, b[i]);

                    wait(b[i]);
                    pc.printf("Waiting for next note... \n\r");

                }
                frequency = 200.0;
                pc.printf("Freqeuncy is reset to 200 Hz.\n\r");
                melody_sum = 0;
                melody_play = 0;
            } else {
                pc.printf("No melody command.");
            }
        medoTimer.stop();
        MEDOT = medoTimer.read_us();
        }
    }
}




//Convert photointerrupter inputs to a rotor state
inline int8_t readRotorState()
{
    return stateMap[I1 + 2*I2 + 4*I3];
}


//Set a given drive state
void motorOut()
{
    curState = readRotorState();
    if(curState == orState) count++;
    //Lookup the output byte from the drive state.
    int8_t driveOut;

    //if(count % 1000 < 500) driveOut = driveTable[((curState-orState+lead+6)%6) & 0x07];
    //else driveOut = driveTable[((curState-orState-lead+6)%6) & 0x07];

    driveOut = driveTable[((curState-orState+lead+6)%6) & 0x07];
    pwm_period = float(1/frequency);


    //Turn off first
    if (~driveOut & 0x01) L1L = 0;
    if (~driveOut & 0x02) L1H = 1;
    if (~driveOut & 0x04) L2L = 0;
    if (~driveOut & 0x08) L2H = 1;
    if (~driveOut & 0x10) L3L = 0;
    if (~driveOut & 0x20) L3H = 1;
    /*
        //Then turn on
        if (driveOut & 0x01) L1L = 1;
        if (driveOut & 0x02) L1H = 0;
        if (driveOut & 0x04) L2L = 1;
        if (driveOut & 0x08) L2H = 0;
        if (driveOut & 0x10) L3L = 1;
        if (driveOut & 0x20) L3H = 0;
    */
    if (driveOut & 0x01) {
        L1L.write((outpwm));
        L1L.period(pwm_period);
    }
    if (driveOut & 0x02) L1H = 0;
    if (driveOut & 0x04) {
        L2L.write((outpwm));
        L2L.period(pwm_period);
    }
    if (driveOut & 0x08) L2H = 0;
    if (driveOut & 0x10) {
        L3L.write((outpwm));
        L3L.period(pwm_period);
    }
    if (driveOut & 0x20) L3H = 0;

}

void motorOut_org(int8_t driveState)
{

    //Lookup the output byte from the drive state.
    int8_t driveOut = driveTable[driveState & 0x07];
    //Turn off first
    if (~driveOut & 0x01) L1L = 0;
    if (~driveOut & 0x02) L1H.write(1);
    if (~driveOut & 0x04) L2L = 0;
    if (~driveOut & 0x08) L2H.write(1);
    if (~driveOut & 0x10) L3L = 0;
    if (~driveOut & 0x20) L3H.write(1);

    //Then turn on
    if (driveOut & 0x01) L1L.write(1);
    if (driveOut & 0x02) L1H = 0;
    if (driveOut & 0x04) L2L.write(1);
    if (driveOut & 0x08) L2H = 0;
    if (driveOut & 0x10) L3L.write(1);
    if (driveOut & 0x20) L3H = 0;
}

void RPS()
{
    photorpsTimer.stop();
    photo_timer = photorpsTimer.read_ms();
    photorpsTimer.reset();
    photorpsTimer.start();
    high_res_RPS = (1000/photo_rotation_time);
    rotation_number++;
}

void ISR()
{
    motorOut();
    if (curState==orState) {
        photo_timer = globalTimer.read_ms();
        photo_rotation_time = abs(photo_timer - old_photo_timer);
        old_photo_timer = photo_timer;
        high_res_RPS = (1000/photo_rotation_time);
        rotation_number++;
        prec_rotation_number = 0;
    }
}

void Precise_ISR()
{
    //motorOut();
    cur_quadrature_state = (CHA_I*2 + CHB_I);
    if(cur_quadrature_state == 2) {

        prec_rotation_number++;
        position_degree = (prec_rotation_number * (360/117)) % 360;
        //position_degree = prec_rotation_number * (360/117);

        if(position_degree > 360.0) {                           //has entered into new rotation
            temp_timer = globalTimer.read_ms();
            one_rotation_time = abs(temp_timer - old_p_timer);
            old_p_timer = temp_timer;
            position_degree -= 360.0;
            //total_prec_rotation++;
        }

        //total_prec_rotation+=position_degree/360;

        if((prec_rotation_number % 117) == 116) {  //This is not robust, requires prec_rotatoin to be 467, although pre_rotation is incremental. but, worst case it misses 4 states, ie 3 degree.
            total_prec_rotation++;
        }
    }
}


//Basic synchronisation routine
int8_t motorHome()
{
    //Put the motor in drive state 0 and wait for it to stabilise
    //Lookup the output byte from the drive state.
    int8_t driveOut = driveTable[0x00 & 0x07];

    //Turn off first
    if (~driveOut & 0x01) L1L = 0;
    if (~driveOut & 0x02) L1H = 1;
    if (~driveOut & 0x04) L2L = 0;
    if (~driveOut & 0x08) L2H = 1;
    if (~driveOut & 0x10) L3L = 0;
    if (~driveOut & 0x20) L3H = 1;

    //Then turn on
    if (driveOut & 0x01) L1L = 1;
    if (driveOut & 0x02) L1H = 0;
    if (driveOut & 0x04) L2L = 1;
    if (driveOut & 0x08) L2H = 0;
    if (driveOut & 0x10) L3L = 1;
    if (driveOut & 0x20) L3H = 0;

    wait(1.0);
    //Get the rotor state
    return readRotorState();
}


void controller_thread()
{

    while(1) {
        //rotation_number, prec_rotation_number,
        if(Current_rotation < (0.9 * TargetRotation_user)) { // compare number of rotation
            controller.setProcessValue(Current_res);
            outpwm = controller.compute();
            //outpwm = 0.9;
        } else {
            dis_to_target = TargetRotation_user - Current_rotation;   //The problem here is to miniise dis_to_target to be just and constantly 0;  this is the same objective as to any PID.
            slow_controller.setProcessValue(dis_to_target);
            outpwm = slow_controller.compute();

            //outpwm = 0.0;
            lead = lead * 0;
        }
    }

}

//Main
int main()
{
    //int8_t orState = 0;    //Rotot offset at motor state 0

    pc.printf("Hello\n\r");

    //Start Global Timer
    globalTimer.start();

    count = 0;
    //Run the motor synchronisation
    orState = motorHome();
    pc.printf("Rotor origin: %x\n\r", orState);
    pc.printf("Default Rotation: %.2f, Velocity: %.2f, Lead: %d \n\r", rev, vel, lead);

    //Interruptions for photointerruptions
    photoInt1.rise(&ISR);
    photoInt2.rise(&ISR);
    photoInt3.rise(&ISR);
    photoInt1.fall(&ISR);
    photoInt2.fall(&ISR);
    photoInt3.fall(&ISR);

    //Interrupts for Encoder
    CHA_I.rise(&Precise_ISR);
    CHB_I.rise(&Precise_ISR);
    CHA_I.fall(&Precise_ISR);
    CHB_I.fall(&Precise_ISR);


    controller.setInputLimits(0.0, 50.0);
    controller.setOutputLimits(0.05, 0.95);
    controller.setBias(0.0);
    controller.setMode(0);  // 0 for manual
    controller.setSetPoint(TargetSpeed_user);  //set rps

    slow_controller.setInputLimits(0.0, abs(TargetRotation_user));  //input = dist to finsih .
    slow_controller.setOutputLimits(0.0, 1.0);
    slow_controller.setBias(0.0);
    slow_controller.setMode(0);  // 0 for manual
    slow_controller.setSetPoint(0.0);

    //Threads

    thread_melody.start(play_melody);



//    int slack = floor(0.30 * TargetSpeed_user);
    outpwm = 1.0;
    motorOut_org(0);
    wait (1.0);
    //motorOut_org(1);
    //motorOut_org(2);

    int org_lead  = lead;
    int old_rev;
    float error = 0.0;

    while(true) {
        
        mainTimer.start();

        read_input_command();
        
        

        TOG = 1;
        TargetRotation_user = rev;  // update revolution

        lead = r*2;                 // update rotate direction
        TargetSpeed_user = vel;     // update velocity
        controller.setSetPoint(TargetSpeed_user);  //set update PID
        //if using photointerup
        Current_rotation = rotation_number;
        Current_res = high_res_RPS;


        //rotation_number, prec_rotation_number,
        if(Current_rotation < (0.8 * abs(TargetRotation_user))) { // compare number of rotation

            controller.setProcessValue(Current_res);
            outpwm = controller.compute();

            //outpwm = 0.9;
        } else {
            dis_to_target = TargetRotation_user - Current_rotation;   //The problem here is to miniise dis_to_target to be just and constantly 0;  this is the same objective as to any PID.

            lead = lead * -1;
            //lead = 0;

            controller.setSetPoint(5.0);
            controller.setProcessValue(Current_res);
            outpwm = controller.compute();

            if(Current_res < 6) {
                lead = lead * -1;
            }

            if(dis_to_target < 4) {
                outpwm = 0.0;
            }

            TOG = 1;
        }
    
        mainTimer.stop();
        
        pc.printf("MEDO: %.3f, main: %.3f, RPS: %.3f, PID_PWM: %f, TOTALrotations = %d, Totalrotations: %d, Degree %.3f, Lead %d, Target: %.2f PWM_Period: %.6f \n\r", MEDOT, mainTimer.read_ms(), high_res_RPS, outpwm, rotation_number, total_prec_rotation, position_degree, lead, TargetSpeed_user, pwm_period);
    }
}