/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "EthernetInterface.h"
#include "AppStatus.h"
#include "lpc_swim_image.h"
#include "lpc_swim_font.h"
#include "BiosLoader.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
#define HEADER_X_OFF            (20-5)
#define HEADER_Y_SPACING        (20-5)
#define ITEM_CAPTION_X_OFF      (40-15)
#define ITEM_CAPTION_Y_SPACING  (swim_get_font_height(_win) + 5)
#define ITEM_VALUE_X_OFF        (140-25)
#define COL3_OFF                (230+70)
#define COL4                    (COL3_OFF + ITEM_VALUE_X_OFF - 5)

#define BIG_HEADER_X_OFF            (25)
#define BIG_HEADER_Y_SPACING        (25)
#define BIG_ITEM_CAPTION_X_OFF      (BIG_HEADER_X_OFF + 10)
#define BIG_ITEM_CAPTION_Y_SPACING  (swim_get_font_height(_win) + 10)
#define BIG_ITEM_VALUE_X_OFF        (160)
#define BIG_COL3_OFF                (520)
#define BIG_COL4                    (BIG_COL3_OFF + BIG_ITEM_VALUE_X_OFF - 5)

#define BTN_OFF    20
 
#define DEMO_VERSION     "v1.0"
#define DEMO_BUILD_INFO  __DATE__ " " __TIME__

#define DESCR(__res, __disp) \
  (((__res) == ((__disp)->currentResolution())) ? ("Active") : (((__disp)->isSupported(__res)) ? ("Supported") : ("N/A")))


/******************************************************************************
 * Global variables
 *****************************************************************************/

extern EthernetInterface eth;
extern bool ethInitialized;
extern bool ethUsingDHCP;

/******************************************************************************
 * Private functions
 *****************************************************************************/

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

void AppStatus::draw()
{
    swim_window_open(_win, 
                     _disp->width(), _disp->height(),         // full size
                     (COLOR_T*)_fb,
                     0,0,_disp->width()-1, _disp->height()-1, // window position and size
                     1,                                       // border
                     BLACK, WHITE, BLACK);                    // colors: pen, backgr, forgr
    

    swim_put_image_xy(_win, _bgImg.pixels, _bgImg.width, _bgImg.height, 
                      (_disp->width()-_bgImg.width)/2, (_disp->height()-_bgImg.height)/2);
    
    if (_disp->width() == 480) {
      draw480x272();
    } else {
      draw800x480();
    }

    _btn = new ImageButton(_win->fb, _win->xpmax - BTN_OFF - _resOk->width(), _win->ypmax - BTN_OFF - _resOk->height(), _resOk->width(), _resOk->height());
    _btn->loadImages(_resOk);
    _btn->draw();
}
    
void AppStatus::draw480x272()
{
    char buff[120];
    
    // Column 1
    swim_put_text_xy(_win, "Demo:", HEADER_X_OFF, HEADER_Y_SPACING);
    swim_put_text_xy(_win, "Version:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
    swim_put_text_xy(_win, "Build:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);
    
    swim_put_text_xy(_win, "Network:", HEADER_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
    swim_put_text_xy(_win, "DHCP/Static:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
    swim_put_text_xy(_win, "IP Address:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);
    swim_put_text_xy(_win, "NetMask:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 7);
    swim_put_text_xy(_win, "Gateway:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);
    swim_put_text_xy(_win, "MAC Address:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 9);

    char mac[6];
    mbed_mac_address(mac);
    snprintf(buff, 19, "%02x:%02x:%02x:%02x:%02x:%02x", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
    swim_put_text_xy(_win, buff, ITEM_CAPTION_X_OFF+10, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 10);

    // Column 2
    swim_put_text_xy(_win, DEMO_VERSION, ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
    swim_put_text_xy(_win, DEMO_BUILD_INFO, ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);

    swim_put_text_xy(_win, ethUsingDHCP?"DHCP":"Static IP", ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
    if (ethInitialized) {
        swim_put_text_xy(_win, eth.get_ip_address(), ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);
        swim_put_text_xy(_win, eth.get_netmask(), ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 7);
        swim_put_text_xy(_win, eth.get_gateway(), ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);
        //swim_put_text_xy(_win, eth.getMACAddress(), ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
    } else {
        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);
        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 7);
        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);    
    }

    // Column 3
    swim_put_text_xy(_win, "Display:", COL3_OFF+HEADER_X_OFF, HEADER_Y_SPACING);
    swim_put_text_xy(_win, "Size:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
    swim_put_text_xy(_win, "Landscape:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);
    swim_put_text_xy(_win, "Resolutions:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 3);
    swim_put_text_xy(_win, "16bit rgb565:", COL3_OFF+ITEM_CAPTION_X_OFF+10, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
    swim_put_text_xy(_win, "18bit rgb666:", COL3_OFF+ITEM_CAPTION_X_OFF+10, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
    swim_put_text_xy(_win, "24bit rgb888:", COL3_OFF+ITEM_CAPTION_X_OFF+10, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);

    swim_put_text_xy(_win, "BIOS:", COL3_OFF+HEADER_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);
    {
        uint8_t type, maj, min, rev;
        BiosLoader::instance().getBiosStats(type, maj, min, rev);
        snprintf(buff, 25, "v%u.%u.%u  (board type %u)", maj, min, rev, type);
        swim_put_text_xy(_win, buff, COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 9);
    }

    // Column 4
    sprintf(buff, "%d x %d pixels", _disp->width(), _disp->height());
    swim_put_text_xy(_win, buff, COL4 -20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
    
    sprintf(buff, "%s", _disp->landscape()?"YES":"NO");
    swim_put_text_xy(_win, buff, COL4 -20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);

    sprintf(buff, "%s", DESCR(Display::Resolution_16bit_rgb565, _disp));
    swim_put_text_xy(_win, buff, COL4, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
    
    sprintf(buff, "%s", DESCR(Display::Resolution_18bit_rgb666, _disp));
    swim_put_text_xy(_win, buff, COL4, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
    
    sprintf(buff, "%s", DESCR(Display::Resolution_24bit_rgb888, _disp));
    swim_put_text_xy(_win, buff, COL4, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);
}

void AppStatus::draw800x480()
{
    char buff[120];
    
    // Column 1
    swim_put_text_xy(_win, "Demo:", BIG_HEADER_X_OFF, BIG_HEADER_Y_SPACING);
    swim_put_text_xy(_win, "Version:", BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 1);
    swim_put_text_xy(_win, "Build:", BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 2);
    
    swim_put_text_xy(_win, "Network:", BIG_HEADER_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 4);
    swim_put_text_xy(_win, "DHCP/Static:", BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 5);
    swim_put_text_xy(_win, "IP Address:", BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 6);
    swim_put_text_xy(_win, "NetMask:", BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 7);
    swim_put_text_xy(_win, "Gateway:", BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 8);
    swim_put_text_xy(_win, "MAC Address:", BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 9);

    // Column 2
    swim_put_text_xy(_win, DEMO_VERSION, BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 1);
    swim_put_text_xy(_win, DEMO_BUILD_INFO, BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 2);

    swim_put_text_xy(_win, ethUsingDHCP?"DHCP":"Static IP", BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 5);
    if (ethInitialized) {
        swim_put_text_xy(_win, eth.get_ip_address(), BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 6);
        swim_put_text_xy(_win, eth.get_netmask(), BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 7);
        swim_put_text_xy(_win, eth.get_gateway(), BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 8);
    } else {
        swim_put_text_xy(_win, "Not Initialized", BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 6);
        swim_put_text_xy(_win, "Not Initialized", BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 7);
        swim_put_text_xy(_win, "Not Initialized", BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 8);    
    }
    char mac[6];
    mbed_mac_address(mac);
    snprintf(buff, 19, "%02x:%02x:%02x:%02x:%02x:%02x", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
    swim_put_text_xy(_win, buff, BIG_ITEM_VALUE_X_OFF-20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 9);

    // Column 3
    swim_put_text_xy(_win, "Display:", BIG_COL3_OFF+BIG_HEADER_X_OFF, BIG_HEADER_Y_SPACING);
    swim_put_text_xy(_win, "Size:", BIG_COL3_OFF+BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 1);
    swim_put_text_xy(_win, "Landscape:", BIG_COL3_OFF+BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 2);
    swim_put_text_xy(_win, "Resolutions:", BIG_COL3_OFF+BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 3);
    swim_put_text_xy(_win, "16bit rgb565:", BIG_COL3_OFF+BIG_ITEM_CAPTION_X_OFF+10, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 4);
    swim_put_text_xy(_win, "18bit rgb666:", BIG_COL3_OFF+BIG_ITEM_CAPTION_X_OFF+10, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 5);
    swim_put_text_xy(_win, "24bit rgb888:", BIG_COL3_OFF+BIG_ITEM_CAPTION_X_OFF+10, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 6);

    swim_put_text_xy(_win, "BIOS:", BIG_COL3_OFF+BIG_HEADER_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 8);
    {
        uint8_t type, maj, min, rev;
        BiosLoader::instance().getBiosStats(type, maj, min, rev);
        snprintf(buff, 25, "v%u.%u.%u  (board type %u)", maj, min, rev, type);
        swim_put_text_xy(_win, buff, BIG_COL3_OFF+BIG_ITEM_CAPTION_X_OFF, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 9);
    }

    // Column 4
    sprintf(buff, "%d x %d pixels", _disp->width(), _disp->height());
    swim_put_text_xy(_win, buff, BIG_COL4 -20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 1);
    
    sprintf(buff, "%s", _disp->landscape()?"YES":"NO");
    swim_put_text_xy(_win, buff, BIG_COL4 -20, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 2);

    sprintf(buff, "%s", DESCR(Display::Resolution_16bit_rgb565, _disp));
    swim_put_text_xy(_win, buff, BIG_COL4, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 4);
    
    sprintf(buff, "%s", DESCR(Display::Resolution_18bit_rgb666, _disp));
    swim_put_text_xy(_win, buff, BIG_COL4, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 5);
    
    sprintf(buff, "%s", DESCR(Display::Resolution_24bit_rgb888, _disp));
    swim_put_text_xy(_win, buff, BIG_COL4, BIG_HEADER_Y_SPACING + BIG_ITEM_CAPTION_Y_SPACING * 6);
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppStatus::AppStatus() : _disp(NULL), _win(NULL), _fb(NULL), _btn(NULL)
{
  _bgImg.pointerToFree = NULL;
  _bgImg.pixels = NULL;
}

AppStatus::~AppStatus()
{
    teardown();
}

bool AppStatus::setup()
{
    RtosLog* log = DMBoard::instance().logger();

    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = _disp->allocateFramebuffer();

    if (_win == NULL || _fb == NULL) {
        log->printf("Failed to allocate memory for framebuffer\r\n");
        return false;
    }
    
    if (Image::decode(_resAbout, Image::RES_16BIT, &_bgImg) != 0) {
        log->printf("Failed to load background image\n");
        return false;
    }
    
    return true;
}

void AppStatus::runToCompletion()
{
    bool done = false;
    draw();
    _btn->setAction(buttonClicked, (uint32_t)&done);
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    // Wait for touches, but the AppLauncher is already listening
    // for new touch event and sends a signal to its thread which
    // is the same as runs this function so it is enough to wait
    // for that signal.
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;
    while(!done) {
      ThisThread::flags_wait_all(0x1);
      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
        if (_btn->handle(coord.x, coord.y, coord.z > 0)) {
            _btn->draw();
        }
      }
    }
    
    // User has clicked the button, restore the original FB
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
}

bool AppStatus::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    if (_btn != NULL) {
        delete _btn;
        _btn = NULL;
    }
    if (_bgImg.pointerToFree != NULL) {
        free(_bgImg.pointerToFree);
        _bgImg.pointerToFree = NULL;
    }    
    return true;
}

void AppStatus::addResource(Resources id, Resource* res)
{
    if (id == Resource_Ok_button) {
        _resOk = res;
    } else {
        _resAbout = res;
    }
}




