/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "EthernetInterface.h"
#include "AppUSBStatus.h"
#include "lpc_swim_image.h"
#include "lpc_swim_font.h"
#include "image_data.h"
#include "BiosLoader.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
#define HEADER_X_OFF            (20-5)
#define HEADER_Y_SPACING        (20-5)
#define ITEM_CAPTION_X_OFF      (40-15)
#define ITEM_CAPTION_Y_SPACING  (swim_get_font_height(_win) + 5)
#define ITEM_VALUE_X_OFF        (140-25)
#define COL3_OFF                (230+70)
#define COL4                    (COL3_OFF + ITEM_VALUE_X_OFF - 5)

#define BTN_OFF    20
 
#define DEMO_VERSION     "v0.9"
#define DEMO_BUILD_INFO  __DATE__ " " __TIME__

#define DESCR(__res, __disp) \
  (((__res) == ((__disp)->currentResolution())) ? ("Active") : (((__disp)->isSupported(__res)) ? ("Supported") : ("N/A")))


static const unsigned char* IMAGES[] = {
  img_usb_masstorage,
  img_usb_masstorage_ejected,
  img_usb_mouse,
  img_usb_mouse_ejected,
  img_usb_keyboard,
  img_usb_keyboard_ejected,
};
static const unsigned int IMAGE_SIZES[] = {
  img_size_usb_masstorage,
  img_size_usb_masstorage_ejected,
  img_size_usb_mouse,
  img_size_usb_mouse_ejected,
  img_size_usb_keyboard,
  img_size_usb_keyboard_ejected,
};
static const char* CAPTIONS[] = {
  "Memory Stick",
  "No Memory Stick",
  "Mouse",
  "No Mouse",
  "Keyboard",
  "No Keyboard",
};

/******************************************************************************
 * Global variables
 *****************************************************************************/

extern bool msdConnected;
extern bool keyboardConnected;
extern bool mouseConnected;

/******************************************************************************
 * Private functions
 *****************************************************************************/

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

void AppUSBStatus::draw()
{
    // Prepare fullscreen
    swim_window_open(_win, 
                   _disp->width(), _disp->height(),         // full size
                   (COLOR_T*)_fb,
                   0,0,_disp->width()-1, _disp->height()-1, // window position and size
                   0,                                       // border
                   BLACK, WHITE, BLACK);                     // colors: pen, backgr, forgr
    //swim_set_title(_win, "USB Status", BLACK);
    
    // Start by drawing all USB Devices disconnected. Each device is drawn in
    // its own window so that the positioning information can be kept here
    int images = NumImages/2;
    int xspace = (_disp->width() - (_images[0].width * images)) / 4;
    int yoff = (_disp->height() - _images[0].height)/3;
    int xoff = xspace;
    for (int i = 1; i < NumImages; i+=2) {
      swim_window_open(&_imageWindows[i/2], 
                       _disp->width(), _disp->height(),         // full size
                       (COLOR_T*)_fb,
                       xoff,yoff,xoff+_images[0].width-1, yoff+_images[0].height+20-1, // window position and size
                       0,                                       // border
                       BLACK, WHITE, BLACK);                     // colors: pen, backgr, forgr
      swim_put_image(&_imageWindows[i/2], _images[i].pixels, _images[i].width, _images[i].height);
        
      swim_put_text_centered_win(&_imageWindows[i/2], CAPTIONS[i], _images[i].height);
      xoff += _images[i].width + xspace;
    }

    _btn = new ImageButton(_win->fb, _win->xpmax - BTN_OFF - _resOk->width(), _win->ypmax - BTN_OFF - _resOk->height(), _resOk->width(), _resOk->height());
    _btn->loadImages(_resOk);
    _btn->draw();
}

void AppUSBStatus::updateStatus(Ids id)    
{
    SWIM_WINDOW_T* pWin = &_imageWindows[id/2];
    swim_clear_screen(pWin, WHITE);
    swim_put_image(pWin, _images[id].pixels, _images[id].width, _images[id].height);    
    swim_put_text_centered_win(pWin, CAPTIONS[id], _images[id].height);
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppUSBStatus::AppUSBStatus() : _disp(NULL), _win(NULL), _fb(NULL), 
  _btn(NULL), _resOk(NULL)
{
  for (int i = 0; i < NumImages; i++) {
    _images[i].pointerToFree = NULL;
    _images[i].pixels = NULL;
  }
}

AppUSBStatus::~AppUSBStatus()
{
    teardown();
}

bool AppUSBStatus::setup()
{
    RtosLog* log = DMBoard::instance().logger();

    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = _disp->allocateFramebuffer();

    if (_win == NULL || _fb == NULL) {
        log->printf("Failed to allocate memory for framebuffer\r\n");
        return false;
    }
    
    for (int i = 0; i < NumImages; i++) {
        if (Image::decode(IMAGES[i], IMAGE_SIZES[i], Image::RES_16BIT, &_images[i]) != 0) {
            log->printf("Failed to load image %d of %d\n", i+1, NumImages);
            return false;
        }
    }
    
    return true;
}

void AppUSBStatus::runToCompletion()
{
    // Alternative 1: use the calling thread's context to run in
    bool done = false;
    draw();
    _btn->setAction(buttonClicked, (uint32_t)&done);
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    // Wait for touches, but the AppLauncher is already listening
    // for new touch event and sends a signal to its thread which
    // is the same as runs this function so it is enough to wait
    // for that signal.
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;
    bool lastMassStorage = false;
    bool lastMouse = false;
    bool lastKeyboard = false;
    while(!done) {
      ThisThread::flags_wait_all_for(0x1, 500);
      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
        if (_btn->handle(coord.x, coord.y, coord.z > 0)) {
            _btn->draw();
        }
      }
      if (lastMassStorage != msdConnected) {
        lastMassStorage = msdConnected;
        updateStatus(lastMassStorage ? USBHost_MassStorage_Inserted : USBHost_MassStorage_Ejected);
      }
      if (lastKeyboard != keyboardConnected) {
        lastKeyboard = keyboardConnected;
        updateStatus(lastKeyboard ? USBHost_Keyboard_Inserted : USBHost_Keyboard_Ejected);
      }
      if (lastMouse != mouseConnected) {
        lastMouse = mouseConnected;
        updateStatus(lastMouse ? USBHost_Mouse_Inserted : USBHost_Mouse_Ejected);
      }
    }
    
    // User has clicked the button, restore the original FB
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
}

bool AppUSBStatus::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    if (_btn != NULL) {
        delete _btn;
        _btn = NULL;
    }
    for (int i = 0; i < NumImages; i++) {
        if (_images[i].pointerToFree != NULL) {
            free(_images[i].pointerToFree);
            _images[i].pointerToFree = NULL;
        }    
    }
    return true;
}


void AppUSBStatus::addResource(Resources id, Resource* res)
{
    _resOk = res;
}

