/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "EthernetInterface.h"
#include "AppUSBStatus.h"
#include "lpc_swim_image.h"
#include "lpc_swim_font.h"
#include "image_data.h"
#include "BiosLoader.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
#define HEADER_X_OFF            (20-5)
#define HEADER_Y_SPACING        (20-5)
#define ITEM_CAPTION_X_OFF      (40-15)
#define ITEM_CAPTION_Y_SPACING  (swim_get_font_height(_win) + 5)
#define ITEM_VALUE_X_OFF        (140-25)
#define COL3_OFF                (230+70)
#define COL4                    (COL3_OFF + ITEM_VALUE_X_OFF - 5)

#define BTN_WIDTH  40
#define BTN_HEIGHT 40
#define BTN_OFF    20
 
#define DEMO_VERSION     "v0.9"
#define DEMO_BUILD_INFO  __DATE__ " " __TIME__

#define DESCR(__res, __disp) \
  (((__res) == ((__disp)->currentResolution())) ? ("Active") : (((__disp)->isSupported(__res)) ? ("Supported") : ("N/A")))


static const unsigned char* IMAGES[] = {
  img_usb_masstorage,
  img_usb_masstorage_ejected,
  img_usb_mouse,
  img_usb_mouse_ejected,
  img_usb_keyboard,
  img_usb_keyboard_ejected,
};
static const unsigned int IMAGE_SIZES[] = {
  img_size_usb_masstorage,
  img_size_usb_masstorage_ejected,
  img_size_usb_mouse,
  img_size_usb_mouse_ejected,
  img_size_usb_keyboard,
  img_size_usb_keyboard_ejected,
};
static const char* CAPTIONS[] = {
  "Memory Stick",
  "No Memory Stick",
  "Mouse",
  "No Mouse",
  "Keyboard",
  "No Keyboard",
};

/******************************************************************************
 * Global variables
 *****************************************************************************/

extern bool msdConnected;
extern bool keyboardConnected;
extern bool mouseConnected;

/******************************************************************************
 * Private functions
 *****************************************************************************/

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

void AppUSBStatus::draw()
{
    // Prepare fullscreen
    swim_window_open(_win, 
                   _disp->width(), _disp->height(),         // full size
                   (COLOR_T*)_fb,
                   0,0,_disp->width()-1, _disp->height()-1, // window position and size
                   0,                                       // border
                   BLACK, WHITE, BLACK);                     // colors: pen, backgr, forgr
    //swim_set_title(_win, "USB Status", BLACK);
    
    // Start by drawing all USB Devices disconnected. Each device is drawn in
    // its own window so that the positioning information can be kept here
    int images = NumImages/2;
    int xspace = (_disp->width() - (_images[0].width * images)) / 4;
    int yoff = (_disp->height() - _images[0].height)/3;
    int xoff = xspace;
    for (int i = 1; i < NumImages; i+=2) {
      swim_window_open(&_imageWindows[i/2], 
                       _disp->width(), _disp->height(),         // full size
                       (COLOR_T*)_fb,
                       xoff,yoff,xoff+_images[0].width-1, yoff+_images[0].height+20-1, // window position and size
                       0,                                       // border
                       BLACK, WHITE, BLACK);                     // colors: pen, backgr, forgr
      swim_put_image(&_imageWindows[i/2], _images[i].pixels, _images[i].width, _images[i].height);
        
      swim_put_text_centered_win(&_imageWindows[i/2], CAPTIONS[i], _images[i].height);
      xoff += _images[i].width + xspace;
    }
//    

//    char buff[120];
//    
//    // Column 1
//    swim_put_text_xy(_win, "Demo:", HEADER_X_OFF, HEADER_Y_SPACING);
//    swim_put_text_xy(_win, "Version:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
//    swim_put_text_xy(_win, "Build:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);

//    swim_put_text_xy(_win, "Network:", HEADER_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
//    swim_put_text_xy(_win, "DHCP/Static:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
//    swim_put_text_xy(_win, "IP Address:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);
//    swim_put_text_xy(_win, "NetMask:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 7);
//    swim_put_text_xy(_win, "Gateway:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);
//    swim_put_text_xy(_win, "MAC Address:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 9);

//    char mac[6];
//    mbed_mac_address(mac);
//    snprintf(buff, 19, "%02x:%02x:%02x:%02x:%02x:%02x", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
//    swim_put_text_xy(_win, buff, ITEM_CAPTION_X_OFF+10, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 10);

//    // Column 2
//    swim_put_text_xy(_win, DEMO_VERSION, ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
//    swim_put_text_xy(_win, DEMO_BUILD_INFO, ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);

//    swim_put_text_xy(_win, ethUsingDHCP?"DHCP":"Static IP", ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
//    if (ethInitialized) {
//        swim_put_text_xy(_win, eth.getIPAddress(), ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);
//        swim_put_text_xy(_win, eth.getNetworkMask(), ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 7);
//        swim_put_text_xy(_win, eth.getGateway(), ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);
//        //swim_put_text_xy(_win, eth.getMACAddress(), ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
//    } else {
//        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);
//        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 7);
//        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF-20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);    
//    }

//    // Column 3
//    swim_put_text_xy(_win, "Display:", COL3_OFF+HEADER_X_OFF, HEADER_Y_SPACING);
//    swim_put_text_xy(_win, "Size:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
//    swim_put_text_xy(_win, "Landscape:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);
//    swim_put_text_xy(_win, "Resolutions:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 3);
//    swim_put_text_xy(_win, "16bit rgb565:", COL3_OFF+ITEM_CAPTION_X_OFF+10, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
//    swim_put_text_xy(_win, "18bit rgb666:", COL3_OFF+ITEM_CAPTION_X_OFF+10, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
//    swim_put_text_xy(_win, "24bit rgb888:", COL3_OFF+ITEM_CAPTION_X_OFF+10, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);

//    swim_put_text_xy(_win, "BIOS:", COL3_OFF+HEADER_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);
//    //swim_put_text_xy(_win, "Info:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);
//    //swim_put_text_xy(_win, "Board Type:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 9);
//    {
//        uint8_t type, maj, min, rev;
//        BiosLoader::instance().getBiosStats(type, maj, min, rev);
//        //snprintf(buff, 19, "v%u.%u.%u", maj, min, rev);
//        //swim_put_text_xy(_win, buff, COL4 -20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 8);
//        //snprintf(buff, 19, "%u", type);
//        //swim_put_text_xy(_win, buff, COL4 -20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 9);
//        snprintf(buff, 25, "v%u.%u.%u  (board type %u)", maj, min, rev, type);
//        swim_put_text_xy(_win, buff, COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 9);
//    }

//    // Column 4
//    sprintf(buff, "%d x %d pixels", _disp->width(), _disp->height());
//    swim_put_text_xy(_win, buff, COL4 -20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
//    
//    sprintf(buff, "%s", _disp->landscape()?"YES":"NO");
//    swim_put_text_xy(_win, buff, COL4 -20, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);

//    //sprintf(buff, "%s %s", _disp->isSupported(Display::Resolution_16bit_rgb565)?"Supported":"N/A", _disp->currentResolution()==Display::Resolution_16bit_rgb565?"  (Active)":"");
//    sprintf(buff, "%s", DESCR(Display::Resolution_16bit_rgb565, _disp));
//    swim_put_text_xy(_win, buff, COL4, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
//    
//    //sprintf(buff, "%s %s", _disp->isSupported(Display::Resolution_18bit_rgb666)?"Supported":"N/A", _disp->currentResolution()==Display::Resolution_18bit_rgb666?"  (Active)":"");
//    sprintf(buff, "%s", DESCR(Display::Resolution_18bit_rgb666, _disp));
//    swim_put_text_xy(_win, buff, COL4, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
//    
//    //sprintf(buff, "%s %s", _disp->isSupported(Display::Resolution_24bit_rgb888)?"Supported":"N/A", _disp->currentResolution()==Display::Resolution_24bit_rgb888?"  (Active)":"");
//    sprintf(buff, "%s", DESCR(Display::Resolution_24bit_rgb888, _disp));
//    swim_put_text_xy(_win, buff, COL4, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);

    _btn = new ImageButton(_win->fb, _win->xpmax - BTN_OFF - BTN_WIDTH, _win->ypmax - BTN_OFF - BTN_HEIGHT, BTN_WIDTH, BTN_HEIGHT);
    _btn->loadImages(img_ok, img_size_ok);
    _btn->draw();
}

void AppUSBStatus::updateStatus(Ids id)    
{
    SWIM_WINDOW_T* pWin = &_imageWindows[id/2];
    swim_clear_screen(pWin, WHITE);
    swim_put_image(pWin, _images[id].pixels, _images[id].width, _images[id].height);    
    swim_put_text_centered_win(pWin, CAPTIONS[id], _images[id].height);
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppUSBStatus::AppUSBStatus() : _disp(NULL), _win(NULL), _fb(NULL), _btn(NULL)
{
  for (int i = 0; i < NumImages; i++) {
    _images[i].pointerToFree = NULL;
    _images[i].pixels = NULL;
  }
}

AppUSBStatus::~AppUSBStatus()
{
    teardown();
}

bool AppUSBStatus::setup()
{
    RtosLog* log = DMBoard::instance().logger();

    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = _disp->allocateFramebuffer();

    if (_win == NULL || _fb == NULL) {
        log->printf("Failed to allocate memory for framebuffer\r\n");
        return false;
    }
    
    for (int i = 0; i < NumImages; i++) {
        if (Image::decode(IMAGES[i], IMAGE_SIZES[i], Image::RES_16BIT, &_images[i]) != 0) {
            log->printf("Failed to load image %d of %d\n", i+1, NumImages);
            return false;
        }
    }
    
    return true;
}

void AppUSBStatus::runToCompletion()
{
    // Alternative 1: use the calling thread's context to run in
    bool done = false;
    draw();
    _btn->setAction(buttonClicked, (uint32_t)&done);
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    // Wait for touches, but the AppLauncher is already listening
    // for new touch event and sends a signal to its thread which
    // is the same as runs this function so it is enough to wait
    // for that signal.
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;
    bool lastMassStorage = false;
    bool lastMouse = false;
    bool lastKeyboard = false;
    while(!done) {
      Thread::signal_wait(0x1, 500);
      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
        if (_btn->handle(coord.x, coord.y, coord.z > 0)) {
            _btn->draw();
        }
      }
      if (lastMassStorage != msdConnected) {
        lastMassStorage = msdConnected;
        updateStatus(lastMassStorage ? USBHost_MassStorage_Inserted : USBHost_MassStorage_Ejected);
      }
      if (lastKeyboard != keyboardConnected) {
        lastKeyboard = keyboardConnected;
        updateStatus(lastKeyboard ? USBHost_Keyboard_Inserted : USBHost_Keyboard_Ejected);
      }
      if (lastMouse != mouseConnected) {
        lastMouse = mouseConnected;
        updateStatus(lastMouse ? USBHost_Mouse_Inserted : USBHost_Mouse_Ejected);
      }
    }
    
    // User has clicked the button, restore the original FB
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
}

bool AppUSBStatus::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    if (_btn != NULL) {
        delete _btn;
        _btn = NULL;
    }
    for (int i = 0; i < NumImages; i++) {
        if (_images[i].pointerToFree != NULL) {
            free(_images[i].pointerToFree);
            _images[i].pointerToFree = NULL;
        }    
    }
    return true;
}



