/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "EthernetInterface.h"
#include "AppStatus.h"
#include "lpc_swim_font.h"
#include "image_data.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
#define HEADER_X_OFF      20
#define HEADER_Y_SPACING  20
#define ITEM_CAPTION_X_OFF      40
#define ITEM_CAPTION_Y_SPACING  (swim_get_font_height(_win) + 5)
#define ITEM_VALUE_X_OFF        140
#define COL3_OFF                230

#define BTN_WIDTH  40
#define BTN_HEIGHT 40
#define BTN_OFF    20
 
/******************************************************************************
 * Global variables
 *****************************************************************************/

#if defined(DM_BOARD_USE_ETHERNET)
  extern EthernetInterface eth;
  extern bool ethInitialized;
  extern bool ethUsingDHCP;
#endif

/******************************************************************************
 * Private functions
 *****************************************************************************/

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

void AppStatus::draw()
{
    // Prepare fullscreen
    swim_window_open(_win, 
                   _disp->width(), _disp->height(),         // full size
                   (COLOR_T*)_fb,
                   0,0,_disp->width()-1, _disp->height()-1, // window position and size
                   1,                                       // border
                   WHITE, BLUE, BLACK);                     // colors: pen, backgr, forgr
    swim_set_title(_win, "Application Status", BLACK);

    char buff[120];
    
    // Column 1
    swim_put_text_xy(_win, "Ethernet:", HEADER_X_OFF, HEADER_Y_SPACING);
    swim_put_text_xy(_win, "IP Address:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
    swim_put_text_xy(_win, "NetMask:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);
    swim_put_text_xy(_win, "Gateway Address:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 3);
    swim_put_text_xy(_win, "MAC Address:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
    swim_put_text_xy(_win, "DHCP/Static IP:", ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);

    // Column 2
#if defined(DM_BOARD_USE_ETHERNET)
    if (ethInitialized) {
        swim_put_text_xy(_win, eth.get_ip_address(), ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
        swim_put_text_xy(_win, eth.get_netmask(), ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);
        swim_put_text_xy(_win, eth.get_gateway(), ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 3);    
        swim_put_text_xy(_win, eth.get_mac_address(), ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
        swim_put_text_xy(_win, ethUsingDHCP?"DHCP":"Static IP", ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
    } else {
#else
    {
#endif
        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);
        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 3);    
        char mac[6];
        mbed_mac_address(mac);
        snprintf(buff, 19, "%02x:%02x:%02x:%02x:%02x:%02x", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
        swim_put_text_xy(_win, buff, ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
        swim_put_text_xy(_win, "Not Initialized", ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
    }

    // Column 3
    swim_put_text_xy(_win, "Display:", COL3_OFF+HEADER_X_OFF, HEADER_Y_SPACING);
    swim_put_text_xy(_win, "Size:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
    swim_put_text_xy(_win, "Resolutions:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 2);
    swim_put_text_xy(_win, "   16bit RGB565:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 3);
    swim_put_text_xy(_win, "   18bit RGB666:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
    swim_put_text_xy(_win, "   24bit RGB888:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
    swim_put_text_xy(_win, "Landscape:", COL3_OFF+ITEM_CAPTION_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);

    // Column 4
    sprintf(buff, "%d x %d pixels", _disp->width(), _disp->height());
    swim_put_text_xy(_win, buff, COL3_OFF+ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 1);
    
    sprintf(buff, "%s %s", _disp->isSupported(Display::Resolution_16bit_rgb565)?"Supported":"N/A", _disp->currentResolution()==Display::Resolution_16bit_rgb565?"  (Active)":"");
    swim_put_text_xy(_win, buff, COL3_OFF+ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 3);
    
    sprintf(buff, "%s %s", _disp->isSupported(Display::Resolution_18bit_rgb666)?"Supported":"N/A", _disp->currentResolution()==Display::Resolution_18bit_rgb666?"  (Active)":"");
    swim_put_text_xy(_win, buff, COL3_OFF+ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 4);
    
    sprintf(buff, "%s %s", _disp->isSupported(Display::Resolution_24bit_rgb888)?"Supported":"N/A", _disp->currentResolution()==Display::Resolution_24bit_rgb888?"  (Active)":"");
    swim_put_text_xy(_win, buff, COL3_OFF+ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 5);
    
    sprintf(buff, "%s", _disp->landscape()?"YES":"NO");
    swim_put_text_xy(_win, buff, COL3_OFF+ITEM_VALUE_X_OFF, HEADER_Y_SPACING + ITEM_CAPTION_Y_SPACING * 6);
    
    _btn = new ImageButton(_win->fb, _win->xpmax - BTN_OFF - BTN_WIDTH, _win->ypmax - BTN_OFF - BTN_HEIGHT, BTN_WIDTH, BTN_HEIGHT);
    _btn->loadImages(img_ok, img_size_ok);
    _btn->draw();
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppStatus::AppStatus() : _disp(NULL), _win(NULL), _fb(NULL), _btn(NULL)
{
}

AppStatus::~AppStatus()
{
    teardown();
}

bool AppStatus::setup()
{
    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = _disp->allocateFramebuffer();

    return (_win != NULL && _fb != NULL);
}

void AppStatus::runToCompletion()
{
    // Alternative 1: use the calling thread's context to run in
    bool done = false;
    draw();
    _btn->setAction(buttonClicked, (uint32_t)&done);
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    // Wait for touches, but the AppLauncher is already listening
    // for new touch event and sends a signal to its thread which
    // is the same as runs this function so it is enough to wait
    // for that signal.
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;
    while(!done) {
      ThisThread::flags_wait_all(0x1);
      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
        if (_btn->handle(coord.x, coord.y, coord.z > 0)) {
            _btn->draw();
        }
      }
    }
    
    // User has clicked the button, restore the original FB
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
}

bool AppStatus::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    if (_btn != NULL) {
        delete _btn;
        _btn = NULL;
    }
    return true;
}


