/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "AppUSBMouse.h"
#include "lpc_swim_font.h"
#include "lpc_colors.h"
#include "USBMouse.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/

/******************************************************************************
 * Local variables
 *****************************************************************************/


/******************************************************************************
 * Private functions
 *****************************************************************************/

void AppUSBMouse::draw()
{
    // Prepare fullscreen
    swim_window_open(_win, 
                   _disp->width(), _disp->height(),         // full size
                   (COLOR_T*)_fb,
                   0,0,_disp->width()-1, _disp->height()-1, // window position and size
                   0,                                       // border
                   BLACK, WHITE, BLACK);                    // colors: pen, backgr, forgr

    swim_put_text_centered_win(_win, "(Move PC's cursor with touch display. Press USER button to exit)", _disp->height()-30);
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppUSBMouse::AppUSBMouse() : _disp(NULL), _win(NULL), _fb(NULL)
{
  _coords[0].z = _coords[1].z = 0;
}

AppUSBMouse::~AppUSBMouse()
{
    teardown();
}

bool AppUSBMouse::setup()
{
    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = _disp->allocateFramebuffer();

    return (_win != NULL && _fb != NULL);
}

void AppUSBMouse::runToCompletion()
{
    bool done = false;
    uint32_t flags = 0;
    draw();
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    DMBoard* board = &DMBoard::instance();
    RtosLog* log = board->logger();
    
    USBMouse mouse;
    
    // Wait for touches
    TouchPanel* touch = DMBoard::instance().touchPanel();
    int set = 0;
    touch_coordinate_t* last = NULL;
    touch_coordinate_t* current = &_coords[set];
    while(!done) {
      // wait for a new touch signal (signal is sent from AppLauncher,
      // which listens for touch events)
      flags = ThisThread::flags_wait_all_for(0x1, 50);
      if (flags == 0) {
        if (board->buttonPressed()) {
          done = true;
        }
      } else {
        if (touch->read(current, 1) == TouchPanel::TouchError_Ok) {
          if (current->z == 0) {
            // lifted the finger so we have to think of the next touch
            // event as the first one (otherwise the relative movement
            // will be wrong)
            last = NULL;
          }
          if (last != NULL) {
            int moveX = current->x - last->x;
            int moveY = current->y - last->y;
            if (moveX != 0 && moveY != 0) {
              mouse.move(current->x - last->x, current->y - last->y);
              //log->printf("{%3d,%3d} -> {%3d, %3d}\n", last->x, last->y, current->x, current->y);
              //log->printf("move(%4d, %4d)\n", moveX, moveY);
            }
          }
          last = current;
          current = &_coords[(++set)&1];
        }
      }
    }
    
    // User has clicked the button, restore the original FB
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
}

bool AppUSBMouse::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    return true;
}
