/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "EthernetInterface.h"
#include "AppNetworkSettings.h"
#include "lpc_swim_font.h"
#include "image_data.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
#define BTN_WIDTH  40
#define BTN_HEIGHT 40
#define BTN_OFF    20

#define ARROW_WIDTH   52
#define ARROW_HEIGHT  52
 
/******************************************************************************
 * Private variables
 *****************************************************************************/

// Ugly but needed for callbacks
static AppNetworkSettings* theApp = NULL;

/******************************************************************************
 * Private functions
 *****************************************************************************/

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

static void fieldClicked(uint32_t x)
{
  if (theApp != NULL) {
    theApp->setActiveField(x);
  }
}

static void increaseValue(uint32_t x)
{
  AppNetworkSettings* app = (AppNetworkSettings*)x;
  app->modifyValue(1);
}

static void decreaseValue(uint32_t x)
{
  AppNetworkSettings* app = (AppNetworkSettings*)x;
  app->modifyValue(-1);
}

static void nextField(uint32_t x)
{
  AppNetworkSettings* app = (AppNetworkSettings*)x;
  app->changeActiveField(true);
}

static void prevField(uint32_t x)
{
  AppNetworkSettings* app = (AppNetworkSettings*)x;
  app->changeActiveField(false);
}

void AppNetworkSettings::draw()
{
    // Prepare fullscreen
    swim_window_open(_win, 
                   _disp->width(), _disp->height(),         // full size
                   (COLOR_T*)_fb,
                   0,0,_disp->width()-1, _disp->height()-1, // window position and size
                   1,                                       // border
                   RED,WHITE,BLACK);/*WHITE, RED,  BLACK);*/                     // colors: pen, backgr, forgr
    swim_set_pen_color(_win, WHITE);
    swim_set_title(_win, "Network Settings", RED);
    swim_set_pen_color(_win, RED);

    //_buttons[ButtonDone] = new Button("Done", _win->fb, _win->xpmax - BTN_OFF - BTN_WIDTH, _win->ypmax - BTN_OFF - BTN_HEIGHT, BTN_WIDTH, BTN_HEIGHT);
  
    ImageButton* ib;
    
    ib =  new ImageButton(_win->fb, _win->xpmax - 2*BTN_OFF - 2*BTN_WIDTH, _win->ypmax - BTN_OFF - BTN_HEIGHT, BTN_WIDTH, BTN_HEIGHT);
    ib->loadImages(img_ok, img_size_ok);
    ib->draw();
    _buttons[ButtonOk] = ib;    
    ib =  new ImageButton(_win->fb, _win->xpmax - BTN_OFF - BTN_WIDTH, _win->ypmax - BTN_OFF - BTN_HEIGHT, BTN_WIDTH, BTN_HEIGHT);
    ib->loadImages(img_cancel, img_size_cancel);
    ib->draw();
    _buttons[ButtonCancel] = ib;    

    ib =  new ImageButton(_win->fb, 300, 40, ARROW_WIDTH, ARROW_HEIGHT);
    ib->loadImages(img_arrow_up, img_size_arrow_up);
    ib->setAction(increaseValue, (uint32_t)this);
    ib->draw();
    _buttons[ButtonUp] = ib;    
    ib =  new ImageButton(_win->fb, 300, 40+ARROW_HEIGHT+ARROW_HEIGHT, ARROW_WIDTH, ARROW_HEIGHT);
    ib->loadImages(img_arrow_down, img_size_arrow_down);
    ib->setAction(decreaseValue, (uint32_t)this);
    ib->draw();
    _buttons[ButtonDown] = ib;
    ib =  new ImageButton(_win->fb, 300-ARROW_WIDTH/2-10, 40+ARROW_HEIGHT, ARROW_WIDTH, ARROW_HEIGHT);
    ib->loadImages(img_arrow_left, img_size_arrow_left);
    ib->setAction(prevField, (uint32_t)this);
    ib->draw();
    _buttons[ButtonLeft] = ib;
    ib =  new ImageButton(_win->fb, 300+ARROW_WIDTH/2+10, 40+ARROW_HEIGHT, ARROW_WIDTH, ARROW_HEIGHT);
    ib->loadImages(img_arrow_right, img_size_arrow_right);
    ib->setAction(nextField, (uint32_t)this);
    ib->draw();
    _buttons[ButtonRight] = ib;
    
    // To avoid having each DigitButton deallocate the shared image
    void* pointerToFree = _digitImage.pointerToFree;
    _digitImage.pointerToFree = NULL;
    
    addIPField(20, ButtonIp0, "IP Address:");
    addIPField(20+65, ButtonMask0, "Net Mask:");
    addIPField(20+65+65, ButtonGw0, "Gateway:");
    
    // Restore shared image so that it will be deallocated during teardown
    _digitImage.pointerToFree = pointerToFree;
    
  for (int i = 0; i < NumButtons; i++) {
    _buttons[i]->draw();
  }
  
  markField(_activeField, true);
}

void AppNetworkSettings::addIPField(int y, int idx, const char* lbl)
{
    DigitButton* db;
    
    swim_put_text_xy(_win, lbl, 10, y);
    y += 15;
    int btny = y-7;
    y += _win->ypvmin; // compensate for title bar 
    int x = 20;
    
    
    db =  new DigitButton(_win->fb, x, y, 55, 34);
    db->loadImages(&_digitImage);//img_numbers, img_size_numbers);
    db->setNumDigits(3);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    
    x += 60;    
    db =  new DigitButton(_win->fb, x, y, 55, 34);
    db->loadImages(&_digitImage);//img_numbers, img_size_numbers);
    db->setNumDigits(3);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    swim_put_box(_win, x-7, btny+31, x-4, btny+34);    
    
    x += 60;    
    db =  new DigitButton(_win->fb, x, y, 55, 34);
    db->loadImages(&_digitImage);//img_numbers, img_size_numbers);
    db->setNumDigits(3);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    swim_put_box(_win, x-7, btny+31, x-4, btny+34);    
    
    x += 60;    
    db =  new DigitButton(_win->fb, x, y, 55, 34);
    db->loadImages(&_digitImage);//img_numbers, img_size_numbers);
    db->setNumDigits(3);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    swim_put_box(_win, x-7, btny+31, x-4, btny+34);    
}

void AppNetworkSettings::markField(int field, bool active)
{
  COLOR_T oldPen = _win->pen;
  COLOR_T oldFill = _win->fill;
  _win->fill = active ? BLACK : _win->bkg;
  _win->pen = active ? BLACK : _win->bkg;
  if (field >= 0 && field < NumFields) {
    int x0, y0, x1, y1;
    _buttons[field]->bounds(x0,y0,x1,y1);
    y1 -= _win->ypvmin+1;
    x0--;
    swim_put_box(_win, x0, y1, x1, y1+3);
  }
  _win->fill = oldFill;
  _win->pen = oldPen;
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppNetworkSettings::AppNetworkSettings() : _disp(NULL), _win(NULL), _fb(NULL), _activeField(0)
{
  for (int i = 0; i < NumButtons; i++) {
    _buttons[i] = NULL;
  }
  _values[0] = 192;
  _values[1] = 168;
  _values[2] = 5;
  _values[3] = 220;
  _values[4] = 255;
  _values[5] = 255;
  _values[6] = 255;
  _values[7] = 0;
  _values[8] = 192;
  _values[9] = 168;
  _values[10] = 5;
  _values[11] = 1;
  
  _digitImage.pointerToFree = NULL;
  
  theApp = this;
}

AppNetworkSettings::~AppNetworkSettings()
{
    theApp = NULL;
    teardown();
}

bool AppNetworkSettings::setup()
{
    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = _disp->allocateFramebuffer();
    int res = Image::decode(img_numbers, img_size_numbers, Image::RES_16BIT, &_digitImage);

    return (_win != NULL && _fb != NULL && res == 0);
}

void AppNetworkSettings::runToCompletion()
{
    // Alternative 1: use the calling thread's context to run in
    bool done = false;
    bool abort = false;
    draw();
    _buttons[ButtonOk]->setAction(buttonClicked, (uint32_t)&done);
    _buttons[ButtonCancel]->setAction(buttonClicked, (uint32_t)&abort);
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    // Wait for touches
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;

    int lastPressed = NumButtons;
    Timer t;
    t.start();
    int repeatAt = 0;

    uint32_t maxDelay = osWaitForever; 
    while(!done && !abort) {
      ThisThread::flags_wait_all(0x1, maxDelay);
      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
        for (int i = 0; i < NumButtons; i++) {
          if (_buttons[i]->handle(coord.x, coord.y, coord.z > 0)) {
            _buttons[i]->draw();
            if (_buttons[i]->pressed()) {
              lastPressed = i; // new button pressed
              t.reset();
              repeatAt = 1000;
            }
          }
        }
        if (lastPressed == ButtonUp || lastPressed == ButtonDown) {
          maxDelay = 10;
          if (_buttons[lastPressed]->pressed() && t.read_ms() > repeatAt) {
            modifyValue((lastPressed == ButtonUp) ? 1 : -1);
            repeatAt = t.read_ms()+(200/(t.read_ms()/1000));
          }
        } else {
          maxDelay = osWaitForever;
        }
      }
    }

    if (!abort) {
#if defined(DM_BOARD_USE_REGISTRY)
      Registry* reg = DMBoard::instance().registry();
      RtosLog* log = DMBoard::instance().logger();
      char buf[16] = {0};
      sprintf(buf, "%u.%u.%u.%u", _values[0], _values[1], _values[2], _values[3]);
      reg->setValue("IP Address", buf);
      log->printf("New STATIC IP Address: %s\n", buf);
      sprintf(buf, "%u.%u.%u.%u", _values[4], _values[5], _values[6], _values[7]);
      reg->setValue("Net Mask", buf);
      log->printf("New STATIC Net Mask:   %s\n", buf);
      sprintf(buf, "%u.%u.%u.%u", _values[8], _values[9], _values[10], _values[11]);
      reg->setValue("Gateway", buf);
      log->printf("New STATIC Gateway:    %s\n", buf);
      reg->store();
#endif
    }
    
    // User has clicked the button, restore the original FB
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
}

bool AppNetworkSettings::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    for (int i = 0; i < NumButtons; i++) {
        if (_buttons[i] != NULL) {
            delete _buttons[i];
            _buttons[i] = NULL;
        }
    }
    if (_digitImage.pointerToFree != NULL) {
        free(_digitImage.pointerToFree);
        _digitImage.pointerToFree = NULL;
    }
    return true;
}

void AppNetworkSettings::modifyValue(int mod)
{
  _values[_activeField] = (mod + _values[_activeField]) % 256;
  ((DigitButton*)_buttons[_activeField])->setValue(_values[_activeField]);
}

void AppNetworkSettings::changeActiveField(bool next)
{
  markField(_activeField, false);
  if (next) {
    _activeField = (_activeField+1) % NumFields;
  } else {
    _activeField = (_activeField+NumFields-1) % NumFields;
  }
  markField(_activeField, true);
}

void AppNetworkSettings::setActiveField(uint32_t newField)
{
  if (_activeField != newField && newField < NumFields) {
    markField(_activeField, false);
    _activeField = newField;
    markField(_activeField, true);
  }
}
