/*
 *  Copyright 2019 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 

/******************************************************************************
 * Includes
 *****************************************************************************/

#include "mbed.h"
#include "dm_rtc.h"
#include "mbed_mktime.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
/******************************************************************************
 * External global variables
 *****************************************************************************/

/******************************************************************************
 * Local variables
 *****************************************************************************/

/******************************************************************************
 * Local Functions
 *****************************************************************************/


/******************************************************************************
 * Public Functions
 *****************************************************************************/ 
 
void dm_init_rtc(void)
{
    LPC_SC->PCONP |= 0x200; // Ensure power is on
    LPC_RTC->CCR = 0x00;
    
    LPC_RTC->CCR |= 1 << 0; // Ensure the RTC is enabled    
}

time_t dm_read_rtc(void)
{
    // Setup a tm structure based on the RTC
    struct tm timeinfo;
    timeinfo.tm_sec = LPC_RTC->SEC;
    timeinfo.tm_min = LPC_RTC->MIN;
    timeinfo.tm_hour = LPC_RTC->HOUR;
    timeinfo.tm_mday = LPC_RTC->DOM;
    timeinfo.tm_mon = LPC_RTC->MONTH - 1;
    timeinfo.tm_year = LPC_RTC->YEAR - 1900;
    
    // Convert to timestamp
    time_t t;
    if (_rtc_maketime(&timeinfo, &t, RTC_4_YEAR_LEAP_YEAR_SUPPORT) == false) {
        return 0;
    }
    
    return t;    
}

void dm_write_rtc(time_t t)
{
    // Convert the time in to a tm
    struct tm timeinfo;
    if (_rtc_localtime(t, &timeinfo, RTC_4_YEAR_LEAP_YEAR_SUPPORT) == false) {
        return;
    }

    // Pause clock, and clear counter register (clears us count)
    LPC_RTC->CCR |= 2;
    
    // Set the RTC
    LPC_RTC->SEC = timeinfo.tm_sec;
    LPC_RTC->MIN = timeinfo.tm_min;
    LPC_RTC->HOUR = timeinfo.tm_hour;
    LPC_RTC->DOM = timeinfo.tm_mday;
    LPC_RTC->MONTH = timeinfo.tm_mon + 1;
    LPC_RTC->YEAR = timeinfo.tm_year + 1900;
    
    // Restart clock
    LPC_RTC->CCR &= ~((uint32_t)2);    
}

int dm_isenabled_rtc(void)
{
    return(((LPC_RTC->CCR) & 0x01) != 0);
}

 