/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "AppSlideShow.h"


/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/

#define TICKER_RESOLUTION_IN_MS  10

#define BTN_OFF    20

/******************************************************************************
 * Global variables
 *****************************************************************************/

volatile uint32_t msTicks = 0;

/******************************************************************************
 * Private functions
 *****************************************************************************/

static void tRender(void const *args)
{
  Renderer* s = (Renderer*)args;
  s->render();
}

static void ticker(void const *n) {
  msTicks += TICKER_RESOLUTION_IN_MS;
}

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppSlideShow::AppSlideShow(const char* scriptFile, const char* pathPrefix, int xoff, int yoff) :
    _fb(NULL), _disp(NULL), _show(NULL), _rend(NULL), 
   _fileMutex(), _btnDone(NULL), _btnRepeat(NULL), _script(NULL), _path(NULL)
{
    _script = (char*)malloc(strlen(scriptFile)+1);
    if (_script != NULL) {
        strcpy(_script, scriptFile);
    }
    if (pathPrefix != NULL) {
        _path = (char*)malloc(strlen(pathPrefix)+1);
        if (_path != NULL) {
            strcpy(_path, pathPrefix);
        }
    }
    _xoff = xoff;
    _yoff = yoff;
}

AppSlideShow::~AppSlideShow()
{
    teardown();
}

bool AppSlideShow::setup()
{
    SlideShow::SlideShowError err;
    
    _disp = DMBoard::instance().display();
    
    _fb = _disp->allocateFramebuffer();
    if (_fb == NULL) {
        err = SlideShow::OutOfMemory;
    } else {
        memset(_fb, 0xff, _disp->fbSize());
        _rend = new Renderer();
        _show = new SlideShow(_rend, _path, _xoff, _yoff, 1, &_fileMutex);
        err = _show->prepare(_script);
    }

    _btnRepeat = new ImageButton((COLOR_T*)_fb, _disp->width() - 2*BTN_OFF - _resOk->width() - _resRepeat->width(), _disp->height() - BTN_OFF - _resRepeat->height(), _resRepeat->width(), _resRepeat->height());
    _btnRepeat->loadImages(_resRepeat);
    _btnDone = new ImageButton((COLOR_T*)_fb, _disp->width() - BTN_OFF - _resOk->width(), _disp->height() - BTN_OFF - _resOk->height(), _resOk->width(), _resOk->height());
    _btnDone->loadImages(_resOk);
    return (err == SlideShow::Ok);
}

void AppSlideShow::runToCompletion()
{
    // Save existing frame buffer
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    bool done = false;
    bool repeat = false;
    _btnDone->setAction(buttonClicked, (uint32_t)&done);
    _btnRepeat->setAction(buttonClicked, (uint32_t)&repeat);
    
    
    // Alternative 1: use the calling thread's context to run in
    //Thread tr(tRender, _rend, osPriorityHigh);
    Thread tr;
    tr.start(callback(tRender, _rend));
    
    // Generate the millisecond ticks for the slideshow
    //RtosTimer rtosTimer(ticker, osTimerPeriodic);
    //rtosTimer.start(TICKER_RESOLUTION_IN_MS);    
    EventQueue rtosTimer(4*EVENTS_EVENT_SIZE);
    rtosTimer.call_every(TICKER_RESOLUTION_IN_MS, ticker, &done);
    rtosTimer.dispatch(0);

    // Wait for touches
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;

    do {
    // Wait for slideshow to complete
    _show->run();
      
      // Temporary ugly way to get current framebuffer
      _btnDone->draw((COLOR_T*)LPC_LCD->UPBASE);
      _btnRepeat->draw((COLOR_T*)LPC_LCD->UPBASE);
       
      done = repeat = false;
      while (!done && !repeat) {
        ThisThread::flags_wait_any(0x1);
        if (touch->read(coord) == TouchPanel::TouchError_Ok) {
          if (_btnDone->handle(coord.x, coord.y, coord.z > 0)) {
            _btnDone->draw();
          }
          if (_btnRepeat->handle(coord.x, coord.y, coord.z > 0)) {
            _btnRepeat->draw();
          }
        }
      }
        
    } while(!done);
    
    tr.terminate();
    
    // Restore the original FB
    _disp->swapFramebuffer(oldFB);
}

bool AppSlideShow::teardown()
{
    if (_show != NULL) {
        free(_show);
        _show = NULL;
    }
    if (_rend != NULL) {
        free(_rend);
        _rend = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    if (_script != NULL) {
        free(_script);
        _script = NULL;
    }
    if (_path != NULL) {
        free(_path);
        _path = NULL;
    }
    return true;
}


void AppSlideShow::addResource(Resources id, Resource* res)
{
    if (id == Resource_Ok_button) {
        _resOk = res;
    } else {
        _resRepeat = res;
    }
}

