/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
#ifndef DIGITBUTTON_H
#define DIGITBUTTON_H

#include "Clickable.h"
#include "Image.h"

/**
 * The DigitButton is used in the same way as the Button so see it for an example
 */
class DigitButton : public Clickable {
public:

    /** Creates a new button
     *
     *  This button will use a SWIM window to draw on. That window will use
     *  part of the full size frame buffer to draw on.
     *
     *  @param fb      the frame buffer
     *  @param x       the upper left corner of the button
     *  @param y       the upper left corner of the button
     *  @param width   the width of the button
     *  @param height  the height of the button
     */
  DigitButton(COLOR_T* fb, uint16_t x, uint16_t y, uint16_t width, uint16_t height);
  virtual ~DigitButton();

    /** Loads the mandatory "normal" state image and the optional "pressed" state image
     *
     *  @param imgUp   the file with the image for the normal state
     *  @param imgDown the file with the image for the pressed state (or NULL to use the same)
     *
     *  @returns
     *       true on success
     *       false on failure
     */
  bool loadImages(const char* imgUp, const char* imgDown = 0);

    /** Loads the mandatory "normal" state image and the optional "pressed" state image
     *
     *  @param imgUp   the decoded image for the normal state
     *  @param imgDown the decoded image for the pressed state (or NULL to use the same)
     *
     *  @returns
     *       true on success
     *       false on failure
     */
  bool loadImages(const Image::ImageData_t* imgUp, const Image::ImageData_t* imgDown = 0);

    /** Loads the mandatory "normal" state image and the optional "pressed" state image
     *
     *  @param imgUp   the image for the normal state
     *  @param imgDown the image for the pressed state (or NULL to use the same)
     *
     *  @returns
     *       true on success
     *       false on failure
     */
  bool loadImages(const unsigned char* imgUp, unsigned int imgUpSize, 
                  const unsigned char* imgDown = 0, unsigned int imgDownSize = 0);
  
  void setNumDigits(unsigned int num);
  void setValue(unsigned int val);

    /** Draws the button (on a new framebuffer if one is specified)
     *  @param fb      the frame buffer
     */
  virtual void draw(COLOR_T* fb = 0);

private:
  Image::ImageData_t _imgUp;
  Image::ImageData_t _imgDown;
  unsigned int _value;
  unsigned int _digits;
  unsigned int _img_y_offset;
  unsigned int _img_digit_height;
  unsigned int _img_digit_width;
  unsigned int _img_digit_size;

  void drawDigits(Image::ImageData_t& img);
};

#endif /* IMAGEBUTTON_H */
