/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#ifndef IMAGE_H
#define IMAGE_H

#include "mbed.h"
#include "rtos.h"

class Resource;

/**
 * Image example
 *
 * @code
 * #include "mbed.h"
 * #include "Image.h"
 *
 * const unsigned char cube_image1[] = { 137,80,78,71, ... };
 * int cube_image1_sz = sizeof(cube_image1);
 *
 * int main(void) {
 *    // initialize the display
 *    ...
 *
 *    // decode an image from an array
 *    Image::ImageData_t img;
 *    if (Image::decode(cube_image1, cube_image1_sz, &img) == 0) {
 *         // draw on display using img.pixels, img.width and img.height
 *         ...
 *         free(img.pixels);
 *    }
 *
 *    // decode an image from a file
 *    if (Image::decode("/ram/image.png", &img) == 0) {
 *         // draw on display using img.pixels, img.width and img.height
 *         ...
 *         free(img.pixels);
 *    }
 * }
 * @endcode
 */
class Image {
public:

    enum Type {
        BMP = 0,
        PNG,
        RAW,      /* Image prepared with the img_conv.jar tool */
        UNKNOWN
    };
    
    enum Resolution {
        RES_16BIT,
        RES_24BIT
    };
    
    typedef struct {
        uint16_t*  pixels;
        uint32_t   width;
        uint32_t   height;
        Resolution res;
        void*      pointerToFree;
    } ImageData_t;

    /** Decodes the specified image data
     *
     *  Note that if this function returns a zero, indicating success,
     *  the pixels member of the pDataOut structure must be
     *  deallocated using free() when no longer needed.
     *
     *  @param pDataIn the image data
     *  @param sizeIn the number of bytes in the pDataIn array
     *  @param res the format of the display
     *  @param pDataOut the decoded image (only valid if 0 is returned)
     *
     *  @returns
     *       0 on success
     *       1 on failure
     */
    static int decode(const unsigned char* pDataIn, unsigned int sizeIn, Resolution res, ImageData_t* pDataOut);

    /** Reads the specified file and decodes the image data
     *
     *  Note that if this function returns a zero, indicating success,
     *  the pixels member of the pDataOut structure must be
     *  deallocated using free() when no longer needed.
     *
     *  @param filename the file name and path
     *  @param res the format of the display
     *  @param pDataOut the decoded image (only valid if 0 is returned)
     *  @param pLock an optional mutex to prevent multiple access to file operations
     *
     *  @returns
     *       0 on success
     *       1 on failure
     */
    static int decode(const char* filename, Resolution res, ImageData_t* pDataOut, Mutex* pLock=NULL);

    /** Decodes the specified image resource
     *
     *  Note that if this function returns a zero, indicating success,
     *  the pixels member of the pDataOut structure must be
     *  deallocated using free() when no longer needed.
     *
     *  The decoded image is cached in the Resource to prevent decoding
     *  the same resource over and over.
     *
     *  @param res the resource to decode
     *  @param resolution the format of the display
     *  @param pDataOut the decoded image (only valid if 0 is returned)
     *
     *  @returns
     *       0 on success
     *       1 on failure
     */
    static int decode(Resource* res, Resolution resolution, ImageData_t* pDataOut, Mutex* pLock=NULL);

private:

    /** No instance needed
     *
     */
    Image();
    
    static Type imageType(const unsigned char* pDataIn, unsigned int sizeIn);

    static uint32_t fileSize(FILE* f);
};

#endif
