/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#include "DigitButton.h"
#include "mbed.h"
#include "DMBoard.h"

#include "lpc_swim_image.h"

DigitButton::DigitButton(COLOR_T* fb, uint16_t x, uint16_t y, uint16_t width, uint16_t height) :
  Clickable(fb, x, y, width, height), _value(0), _digits(1), _img_y_offset(0),
  _img_digit_height(0), _img_digit_width(0), _img_digit_size(0)
{
  _imgUp.pointerToFree = NULL;
  _imgUp.pixels = NULL;
  _imgDown.pointerToFree = NULL;
  _imgDown.pixels = NULL;
  _win.fill = WHITE;
  swim_clear_screen(&_win, _win.fill);
}

DigitButton::~DigitButton()
{
  if (_imgUp.pointerToFree != NULL) {
    free(_imgUp.pointerToFree);
    _imgUp.pointerToFree = NULL;
  }
  if (_imgDown.pointerToFree != NULL) {
    free(_imgDown.pointerToFree);
    _imgDown.pointerToFree = NULL;
  }
}

bool DigitButton::loadImages(const char* imgUp, const char* imgDown)
{
  if (_imgUp.pointerToFree != NULL) {
    free(_imgUp.pointerToFree);
    _imgUp.pointerToFree = NULL;
  }
  if (_imgDown.pointerToFree != NULL) {
    free(_imgDown.pointerToFree);
    _imgDown.pointerToFree = NULL;
  }
  if (Image::decode(imgUp, Image::RES_16BIT, &_imgUp) != 0) {
    DMBoard::instance().logger()->printf("Failed to load %s\n", imgUp);
    return false;
  }
  if (imgDown != NULL) {
    if (Image::decode(imgDown, Image::RES_16BIT, &_imgDown) == 0) {
      DMBoard::instance().logger()->printf("Failed to load %s\n", imgDown);
      return false;
    }
  }
  _img_digit_width = _imgUp.width;
  _img_digit_height = _imgUp.height/10;
  _img_y_offset = _imgUp.height - 10*_img_digit_height;
  _img_digit_size = ((_imgUp.res == Image::RES_16BIT)?1:2) * _img_digit_height * _img_digit_width;
  return true;
}

bool DigitButton::loadImages(const Image::ImageData_t* imgUp, const Image::ImageData_t* imgDown)
{
  if (_imgUp.pointerToFree != NULL) {
    free(_imgUp.pointerToFree);
    _imgUp.pointerToFree = NULL;
  }
  if (_imgDown.pointerToFree != NULL) {
    free(_imgDown.pointerToFree);
    _imgDown.pointerToFree = NULL;
  }
  memcpy(&_imgUp, imgUp, sizeof(Image::ImageData_t));
  if (imgDown != NULL) {
    memcpy(&_imgDown, imgDown, sizeof(Image::ImageData_t));
  }
  _img_digit_width = _imgUp.width;
  _img_digit_height = _imgUp.height/10;
  _img_y_offset = _imgUp.height - 10*_img_digit_height;
  _img_digit_size = ((_imgUp.res == Image::RES_16BIT)?1:2) * _img_digit_height * _img_digit_width;
  return true;
}

bool DigitButton::loadImages(const unsigned char* imgUp, unsigned int imgUpSize, 
                             const unsigned char* imgDown, unsigned int imgDownSize)
{
  if (_imgUp.pointerToFree != NULL) {
    free(_imgUp.pointerToFree);
    _imgUp.pointerToFree = NULL;
  }
  if (_imgDown.pointerToFree != NULL) {
    free(_imgDown.pointerToFree);
    _imgDown.pointerToFree = NULL;
  }
  if (Image::decode(imgUp, imgUpSize, Image::RES_16BIT, &_imgUp) != 0) {
    DMBoard::instance().logger()->printf("Failed to load %s\n", imgUp);
    return false;
  }
  if (imgDown != NULL) {
    if (Image::decode(imgDown, imgDownSize, Image::RES_16BIT, &_imgDown) == 0) {
      DMBoard::instance().logger()->printf("Failed to load %s\n", imgDown);
      return false;
    }
  }
  _img_digit_width = _imgUp.width;
  _img_digit_height = _imgUp.height/10;
  _img_y_offset = _imgUp.height - 10*_img_digit_height;
  _img_digit_size = ((_imgUp.res == Image::RES_16BIT)?1:2) * _img_digit_height * _img_digit_width;
  return true;
}

void DigitButton::setNumDigits(unsigned int num)
{
  _digits = num;
  draw();
}

void DigitButton::setValue(unsigned int val)
{
  _value = val;
  draw();
}

void DigitButton::draw(COLOR_T* fb)
{
  if (fb != NULL) {
    _win.fb = fb;
  }
//  if (_pressed) {
//    if (_imgDown.pixels != NULL) {
//      swim_put_image(&_win, _imgDown.pixels, _imgDown.width, _imgDown.height);
//    }
//  } else {
//    if (_imgUp.pixels != NULL) {
//      swim_put_image(&_win, _imgUp.pixels, _imgUp.width, _imgUp.height);
//    }
//  }
  drawDigits(_pressed ? _imgDown : _imgUp);
}

void DigitButton::drawDigits(Image::ImageData_t& img)
{
  if (img.pixels != NULL) {
    uint32_t v = _value;
    int x = (_win.xvsize - (_digits*_img_digit_width))/2;
    if (x < 0) {
      x = 0;
    }
    int y = (_win.yvsize - _img_digit_height)/2;
    for (unsigned int i = 0; i < _digits; i++) {
      uint32_t off = (v % 10)*_img_digit_size + _img_y_offset*_img_digit_width;
      swim_put_image_xy(&_win, _imgUp.pixels+off, _img_digit_width, _img_digit_height, x+(_digits-i-1)*_img_digit_width, y);
      v = v/10;
    }
  }
}
