/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
#ifndef CLICKABLE_H
#define CLICKABLE_H

#include "lpc_swim.h"

/**
 * Clickable is an abstract base class for the Button and ImageButton.
 */
class Clickable {
public:

    /** Creates a new clickable
     *
     *  This clickable will use a SWIM window to draw on. That window will use
     *  part of the full size frame buffer to draw on.
     *
     *  @param fb      the frame buffer
     *  @param x       the upper left corner of the button
     *  @param y       the upper left corner of the button
     *  @param width   the width of the button
     *  @param height  the height of the button
     */
  Clickable(COLOR_T* fb, uint16_t x, uint16_t y, uint16_t width, uint16_t height);
  virtual ~Clickable() {};
  
    /** Set the function to call when  clicked
     *
     *  Note that this function can be called with NULL as func to unregister the
     *  callback function.
     *
     *  @param func the function to call
     *  @param arc the argument to pass to the function when calling
     */
  void setAction(void (*func)(uint32_t arg), uint32_t arg) { _func = func; _funcArg = arg; }
  
    /** Process the touch event
     *
     *  This function will detect if and how the touch event affects it.
     *  If the event causes a click then the registered
     *  callback function is called before handle() returns.
     *
     *  The return value is to let the caller now if the button should be
     *  redrawn or not.
     *
     *  @param x       the touched x coordinate
     *  @param y       the touched y coordinate
     *  @param pressed true if the user pressed the display
     *
     *  @returns
     *       true if the button should be redrawn
     *       false if the event did not affect the button
     */
  bool handle(uint16_t x, uint16_t y, bool pressed);

    /** Test if the button is held down (usable for repeated presses)
     *
     *  @returns
     *       true if the button is pressed
     *       false otherwise
     */
  bool pressed() { return _pressed; }
  
  void bounds(int& x0, int& y0, int&x1, int&y1) {
      x0 = _win.xpmin;
      y0 = _win.ypmin;
      x1 = _win.xpmax;
      y1 = _win.ypmax;
  }

    /** Draws the button (on a new framebuffer if one is specified)
     *  @param fb      the frame buffer
     */
  virtual void draw(COLOR_T* fb = 0) = 0;
  
protected:
  bool _enabled, _pressed;
  SWIM_WINDOW_T _win;

private:
  void (*_func)(uint32_t arg);
  uint32_t _funcArg;
};

#endif /* CLICKABLE_H */
