/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#include "ImageButton.h"
#include "mbed.h"
#include "DMBoard.h"

#include "lpc_swim_image.h"
#include "lpc_swim_font.h"
#include "lpc_colors.h"

ImageButton::ImageButton(COLOR_T* fb, uint16_t x, uint16_t y, uint16_t width, uint16_t height,
                         const char* caption, COLOR_T color) :
  Clickable(fb, x, y, width, height+(caption==NULL?0:20))
{
  _imgUp.pointerToFree = NULL;
  _imgUp.pixels = NULL;
  _imgDown.pointerToFree = NULL;
  _imgDown.pixels = NULL;
  _caption = NULL;
  _transparent = false;
    
  if (caption != NULL) {
    _caption = (char*)malloc(strlen(caption)+1);
    if (_caption != NULL) {
      strcpy(_caption, caption);
    }
    _captionColor = color;
  }
}

ImageButton::~ImageButton()
{
  if (_imgUp.pointerToFree != NULL) {
    free(_imgUp.pointerToFree);
    _imgUp.pointerToFree = NULL;
  }
  if (_imgDown.pointerToFree != NULL) {
    free(_imgDown.pointerToFree);
    _imgDown.pointerToFree = NULL;
  }
  if (_caption != NULL) {
    free(_caption);
    _caption = NULL;
  }
}

bool ImageButton::loadImages(const char* imgUp, const char* imgDown)
{
  if (_imgUp.pointerToFree != NULL) {
    free(_imgUp.pointerToFree);
    _imgUp.pointerToFree = NULL;
  }
  if (_imgDown.pointerToFree != NULL) {
    free(_imgDown.pointerToFree);
    _imgDown.pointerToFree = NULL;
  }
  if (Image::decode(imgUp, Image::RES_16BIT, &_imgUp) != 0) {
    DMBoard::instance().logger()->printf("Failed to load %s\n", imgUp);
    return false;
  }
  if (imgDown != NULL) {
    if (Image::decode(imgDown, Image::RES_16BIT, &_imgDown) == 0) {
      DMBoard::instance().logger()->printf("Failed to load %s\n", imgDown);
      return false;
    }
  }
  return true;
}

bool ImageButton::loadImages(const unsigned char* imgUp, unsigned int imgUpSize, 
                             const unsigned char* imgDown, unsigned int imgDownSize)
{
  if (_imgUp.pointerToFree != NULL) {
    free(_imgUp.pointerToFree);
    _imgUp.pointerToFree = NULL;
  }
  if (_imgDown.pointerToFree != NULL) {
    free(_imgDown.pointerToFree);
    _imgDown.pointerToFree = NULL;
  }
  if (Image::decode(imgUp, imgUpSize, Image::RES_16BIT, &_imgUp) != 0) {
    DMBoard::instance().logger()->printf("Failed to load %s\n", imgUp);
    return false;
  }
  if (imgDown != NULL) {
    if (Image::decode(imgDown, imgDownSize, Image::RES_16BIT, &_imgDown) == 0) {
      DMBoard::instance().logger()->printf("Failed to load %s\n", imgDown);
      return false;
    }
  }
  return true;
}

bool ImageButton::loadImages(Resource* resImgUp, Resource* resImgDown)
{
  if (_imgUp.pointerToFree != NULL) {
    free(_imgUp.pointerToFree);
    _imgUp.pointerToFree = NULL;
  }
  if (_imgDown.pointerToFree != NULL) {
    free(_imgDown.pointerToFree);
    _imgDown.pointerToFree = NULL;
  }
  if (Image::decode(resImgUp, Image::RES_16BIT, &_imgUp) != 0) {
    DMBoard::instance().logger()->printf("Failed to load \"up\" image\n");
    return false;
  }
  if (resImgDown != NULL) {
    if (Image::decode(resImgDown, Image::RES_16BIT, &_imgDown) == 0) {
      DMBoard::instance().logger()->printf("Failed to load \"down\" image\n");
      return false;
    }
  }
  return true;
}


void ImageButton::setTransparency(COLOR_T tColor)
{
  _transparent = true;
  _transparentColor = tColor;
}

void ImageButton::draw(COLOR_T* fb)
{
  if (fb != NULL) {
    _win.fb = fb;
  }
  if (_caption != NULL) {
    _win.pen = _captionColor;
    swim_put_text_centered_win(&_win, _caption, _imgUp.height+2);
  }
  if (_pressed) {
    if (_imgDown.pixels != NULL) {
      if (_transparent) {
        swim_put_transparent_image_xy(&_win, _imgDown.pixels, _imgDown.width, _imgDown.height, 0, 0, _transparentColor);
      } else {
        swim_put_image(&_win, _imgDown.pixels, _imgDown.width, _imgDown.height);
      }
    }
  } else {
    if (_imgUp.pixels != NULL) {
      if (_transparent) {
        swim_put_transparent_image_xy(&_win, _imgUp.pixels, _imgUp.width, _imgUp.height, 0, 0, _transparentColor);
      } else {
        swim_put_image(&_win, _imgUp.pixels, _imgUp.width, _imgUp.height);
      }
    }
  }
}
