/*
 * wave.cpp -- wave templates for use in Digital 
 *    Theremin demo program 
 */
#include "mbed.h"
#include "dma.h"
#include "jswitch.h"
#include "wave.h"

// Default wave type is the SAW
static int wavetype = WAVE_TYPE_DEFAULT;
// Next wave type requested
static bool next_wavetype=false;
// The wave template buffer size is DMA_BUFSIZE
static int waveform[1+WAVE_TYPE_LAST - WAVE_TYPE_FIRST][DMA_BUFSIZE];
// Accumulated phase
static unsigned accum_phi = 0;

/*
 * the phase accumulator ranges between 0 and 99999. 
 * corresponding to 0 and 2*pi radians.  The frequency 
 * supplied to the wave_nextval() function determines the 
 * magnitude of the  change in the phase accumulator value, 
 * according to the formula: 
 *       delta-phase = 100000*frequency/WAVE_SAMPLE_RATE
 * The next value of phase_accumulator is: 
 *       (phase_accumulator + delta-phase) % 100000
 * phase_accumulator is converted to wave table index using 
 * the formula: 
 *        index = (DMA_BUFSIZE - 1)*phase_accumulator/0xFFFF
 */

void wave_init(void)
{
    int j;

    accum_phi = 0;
    wavetype = WAVE_TYPE_DEFAULT;

    for (j=0;j<DMA_BUFSIZE;j++) {
        waveform[WAVE_TYPE_SAW][j] = j;
    }

    for (j=0;j<DMA_BUFSIZE;j++) {
        if (j<DMA_BUFSIZE/2) {
            waveform[WAVE_TYPE_SQUARE][j] = DMA_BUFSIZE; 
        } else {
            waveform[WAVE_TYPE_SQUARE][j] = 0;
        }
    }

    for (j=0;j<DMA_BUFSIZE;j++) {
        if (j<DMA_BUFSIZE/2) {
            waveform[WAVE_TYPE_TRIANGLE][j] = 2*j;
        } else {
            waveform[WAVE_TYPE_TRIANGLE][j]
                =DMA_BUFSIZE-2*(j-DMA_BUFSIZE/2);
        }
    }
}

void wave_reset(void)
{
    accum_phi=0;
}
/* 
 * wave_nextval 
 *       delta-phase = 100000*frequency/WAVE_SAMPLE_RATE
 * The next value of phase_accumulator is: 
 *       (phase_accumulator + delta-phase) % 100000
 * phase_accumulator is converted to wave table index using 
 * the formula: 
 *        index = DMA_BUFSIZE * phase_accumulator / 100000 
 */
int wave_nextval(unsigned freq)
{
    unsigned delta_phi,index;

    delta_phi= 100000*freq/WAVE_SAMPLE_RATE;
    accum_phi = (accum_phi + delta_phi)%100000;
    index = (DMA_BUFSIZE * accum_phi)/100000;
    return waveform[wavetype][index];
}

void wave_update(void)
{
    unsigned char js_val;

    js_val = js_read(JS_CENTER);
    if ((js_val & JS_CENTER)==JS_CENTER) {
        next_wavetype = true;
    }
}

bool wave_type_changed(void)
{
    return next_wavetype;
}

void wave_type_incr(void)
{
    next_wavetype = false;
    wavetype = 1+wavetype;
    if (wavetype > WAVE_TYPE_LAST) {
        wavetype = WAVE_TYPE_FIRST;
    }
}

