/*
 * envlp.c -- manage the note's envelope state machine
 */
#include "mbed.h"
#include "debug.h"
#include "touch.h"
#include "jswitch.h"
#include "envlp.h"
/*
 * sound envelope follows ADSR pattern 
 * (Attack, Decay, Sustain, Release) 
 * three parameters below control the envelope shape 
 * It should be easy to consider these default values 
 * and modify the code below to set the variables 
 * envlp_attack_bufs, envlp_release_bufs, 
 * and envlp_release_delta dynamically, 
 * depending upon external inputs 
 */ 
#define ATTACK_BUFFERS_MIN  2
#define ATTACK_BUFFERS_MAX 20
#define ATTACK_BUFFERS 5     // attack buffers
#define DECAY_BUFFERS_MIN 1
#define DECAY_BUFFERS_MAX 40
#define DECAY_BUFFERS  ATTACK_BUFFERS/2 // decay buffers
#define RELEASE_DELTA_MIN 1
#define RELEASE_DELTA_MAX 128
#define RELEASE_DELTA  8    // inverse release duration param

/*
 * local envelope functions
 */
static void envlp_set_attack_bufs(void);
static void envlp_set_decay_bufs(void);
static void envlp_set_release_delta(void);

/*
 * local envelop veriables
 */
static int envlp_attack_bufs;
static int envlp_decay_bufs;
static int envlp_release_delta;

void envlp_init(void)
{
    envlp_attack_bufs=ATTACK_BUFFERS;  
    envlp_decay_bufs=DECAY_BUFFERS;  
    envlp_release_delta=RELEASE_DELTA;
}

void envlp_update(void)
{
    envlp_set_attack_bufs();
    envlp_set_decay_bufs();
    envlp_set_release_delta();
}

void envlp_set_attack_bufs()
{
    unsigned char js_val;

    js_val = js_read((JS_UP | JS_DOWN));
    if (js_val & JS_UP) {
        if (envlp_attack_bufs < ATTACK_BUFFERS_MAX) {
            envlp_attack_bufs++;
        }
    } else if(js_val & JS_DOWN) {
        if (envlp_attack_bufs > ATTACK_BUFFERS_MIN) {
            envlp_attack_bufs--;
        }
    }
}

int envlp_get_attack_bufs(void)
{
    return envlp_attack_bufs;
}

void envlp_set_decay_bufs(void)
{
    if (envlp_attack_bufs>1) {
        envlp_decay_bufs = envlp_attack_bufs/2;
    } else {
        envlp_decay_bufs = 1;
    }
}

int envlp_get_decay_bufs(void)
{
    return envlp_decay_bufs;
}

void envlp_set_release_delta(void)
{
    unsigned char js_val;

    js_val = js_read((JS_LEFT | JS_RIGHT));
    if (js_val & JS_LEFT) {
        if (envlp_release_delta < RELEASE_DELTA_MAX) {
            envlp_release_delta++;
        }
    } else if(js_val & JS_RIGHT) {
        if (envlp_release_delta > RELEASE_DELTA_MIN) {
            envlp_release_delta--;
        }
    }
}

int envlp_get_release_delta(void)
{
    return envlp_release_delta;
}
