/** Stepper Motor (Unipolar) control library
 *
 *  @class   StepperMotorUni
 *  @author  Dwijay.Edutech Learning Solutions
 *  @version 1.0
 *  @date    1-Feb-2016
 *
 *  The library that controls stepper motor via motor driver chip
 *  This is a driver for a unipolar stepper motor.
 *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "StepperMotorUni.h"
 *
 *  StepperMotorUni motor( p26, p25, p24, p23 );
 *
 *  int main()
 *  {
 *      motor.set_operation_mode(StepperMotorUni::STEP);
 *
 *      while ( 1 ) {
 *          motor.rotate_angle(StepperMotorUni::CLOCKWISE,90,0.02);
 *          wait( 1 );
 *
 *          motor.rotate_angle(StepperMotorUni::COUNTER_CLOCKWISE,90,0.02);
 *          wait( 1 );
 *      }
 *  }
 *  @endcode
 */

#include "mbed.h"
#include "StepperMotorUni.h"


StepperMotorUni::StepperMotorUni(
    PinName out_A,
    PinName out_B,
    PinName out_C,
    PinName out_D
) : motor_out( out_A, out_B, out_C, out_D ),
    rot_mode( CLOCKWISE ),
    sync_mode( ASYNCHRONOUS ),
    phase_mode(STEP),
    current_pos( 0 ),
    max_msps( MAX_MSPS ),
    target_pos( 0 ),
    pause( true )
{
    msps     = max_msps;
    pattern = (unsigned char *)pattern_step_cw;
    pat_index_mask  = 0x3;
//    t.attach( this, &StepperMotorUni::motor_maintain, (float)msps );
}


void StepperMotorUni::motor_maintain( void )
{
    if ( pause )
        return;

    current_pos = current_pos + ( (target_pos < current_pos) ? -1 : 1 );
    send_sequence(current_pos);

//    printf( "%d>>>%d\r\n", current_pos, target_pos );

    if ( target_pos == current_pos ) {
        current_pos = 0;
        target_pos  = 0;
        set_pause(true);
        t.detach();
    }
};


void StepperMotorUni::send_sequence(int stepPos)
{
    motor_out = pattern[stepPos & pat_index_mask];
}

void StepperMotorUni::set_operation_mode( OperationMode v )
{
    phase_mode  = v;
}

void StepperMotorUni::set_sync_mode( SyncMode m )
{
    sync_mode    = m;
}

int StepperMotorUni::distance( void )
{
    return( target_pos - current_pos );
}

void StepperMotorUni::set_target_pos( int p )
{
    target_pos  = p;

    if (sync_mode == SYNCHRONOUS)
        while ( distance() )
            wait( 0 );
}

void StepperMotorUni::set_pause( int sw )
{
    pause   = sw;
}

void StepperMotorUni::stop( void )
{
    target_pos  = current_pos;
    set_pause(true);
    t.detach();
}


void StepperMotorUni::rotate_angle(RotMode StMotorDirection, int Angle, float Speed)
{
    switch ( phase_mode ) {
        case StepperMotorUni::STEP :
            if(StMotorDirection == CLOCKWISE){
                pattern = pattern_step_cw;
            }else if (StMotorDirection == COUNTER_CLOCKWISE) {
                pattern = pattern_step_acw;
            }
            pat_index_mask  = 0x3;
            set_target_pos((int)(Angle/CAL_ANGLE));
            break;

        case StepperMotorUni::HALFSTEP :
            if(StMotorDirection == CLOCKWISE){
                pattern = pattern_halfstep_cw;
            }else if (StMotorDirection == COUNTER_CLOCKWISE) {
                pattern = pattern_halfstep_acw;
            }
            pat_index_mask  = 0x7;
            set_target_pos((int)(2*Angle/CAL_ANGLE));
            break;

        default:
            break;
    }

    current_pos = 0;

//    printf("target pos %d",target_pos);
    set_pause(false);

    msps = Speed;

    t.detach();
    t.attach( this, &StepperMotorUni::motor_maintain, (float)msps );
}


void StepperMotorUni::rotate_steps(RotMode StMotorDirection, int Steps, float Speed)
{
    switch ( phase_mode ) {
        case StepperMotorUni::STEP :
            if(StMotorDirection == CLOCKWISE){
                pattern = pattern_step_cw;
            }else if (StMotorDirection == COUNTER_CLOCKWISE) {
                pattern = pattern_step_acw;
            }
            pat_index_mask  = 0x3;
            break;

        case StepperMotorUni::HALFSTEP :
            if(StMotorDirection == CLOCKWISE){
                pattern = pattern_halfstep_cw;
            }else if (StMotorDirection == COUNTER_CLOCKWISE) {
                pattern = pattern_halfstep_acw;
            }
            pat_index_mask  = 0x7;
            break;

        default:
            break;
    }

    set_target_pos(Steps);
    current_pos = 0;

//    printf("target pos %d",target_pos);
    set_pause(false);

    msps = Speed;

    t.detach();
    t.attach( this, &StepperMotorUni::motor_maintain, (float)msps );
}


#if GENERAL
unsigned char StepperMotorUni::pattern_one_phase[ 4 ]  = { 0x1, 0x2, 0x4, 0x8 };
unsigned char StepperMotorUni::pattern_two_phase[ 4 ]  = { 0x3, 0x6, 0xC, 0x9 };
unsigned char StepperMotorUni::pattern_halfstep[ 8 ]  = { 0x1, 0x3, 0x2, 0x6, 0x4, 0xC, 0x8, 0x9 };
#endif
#if STM601
unsigned char StepperMotorUni::pattern_step_cw[ 4 ]  = { 0xA, 0x3, 0x5, 0xC };
unsigned char StepperMotorUni::pattern_step_acw[ 4 ]  = { 0xC, 0x5, 0x3, 0xA };
unsigned char StepperMotorUni::pattern_halfstep_cw[ 8 ]  = { 0xC, 0x8, 0xA, 0x2, 0x3, 0x1, 0x5, 0x4 };
unsigned char StepperMotorUni::pattern_halfstep_acw[ 8 ]  = { 0x4, 0x5, 0x1, 0x3, 0x2, 0xA, 0x8, 0xC };
#endif
