/** Stepper Motor (Unipolar) control library
 *
 *  @class   StepperMotorUni
 *  @author  Dwijay.Edutech Learning Solutions
 *  @version 1.0
 *  @date    1-Feb-2016
 *
 *  The library that controls stepper motor via motor driver chip
 *  This is a driver for a unipolar stepper motor.
 *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "StepperMotorUni.h"
 *
 *  StepperMotorUni motor( p26, p25, p24, p23 );
 *
 *  int main()
 *  {
 *      motor.set_operation_mode(StepperMotorUni::STEP);
 *
 *      while ( 1 ) {
 *          motor.rotate_angle(StepperMotorUni::CLOCKWISE,90,0.02);
 *          wait( 1 );
 *
 *          motor.rotate_angle(StepperMotorUni::COUNTER_CLOCKWISE,90,0.02);
 *          wait( 1 );
 *      }
 *  }
 *  @endcode
 */

#ifndef    MBED_STEPPERMOTOR_UNIPOLAR
#define    MBED_STEPPERMOTOR_UNIPOLAR

#include "mbed.h"

/* Default Value */
#define MAX_MSPS   0.5   // 500 millisecond per step

/******************************************************************************/
/*                      Stepper Motor Calibration                             */
/******************************************************************************/
#define     CAL_ANGLE           1.8         // Stepper motor calibration angle


/******************************************************************************/
/*                      Stepper Motor Selection                             */
/******************************************************************************/
#define     GENERAL     false
#define     STM601      true

class StepperMotorUni
{
public:

    /** Constants for motor rotate mode */
    typedef enum  {
        STEP,                       /**< Single step operation (default)    */
        HALFSTEP                    /**< half step operation             */
    } OperationMode;

    /** Constants for motor rotate mode */
    typedef enum  {
        CLOCKWISE,             /**< one-way: clockwise turn                */
        COUNTER_CLOCKWISE      /**< one-way: counter clockwise turn        */
    } RotMode;

    /** Constants for syncronization mode */
    typedef enum  {
        ASYNCHRONOUS,               /**< program does wait motor turn completion (default)  */
        SYNCHRONOUS                 /**< program doesn't wait motor turn completion         */
    } SyncMode;

    /** Create a stepper motor object connected to specified DigitalOut pins and a DigitalIn pin
     *
     *  @param out_A DigitalOut pin for motor pulse signal-A
     *  @param out_B DigitalOut pin for motor pulse signal-B
     *  @param out_C DigitalOut pin for motor pulse signal-C
     *  @param out_D DigitalOut pin for motor pulse signal-D
     */
    StepperMotorUni(
        PinName out_A,
        PinName out_B,
        PinName out_C,
        PinName out_D
    );

    /**
     * @brief Sends Sequence to turn motor
     * @param stepPos pattern index value/ sequence number
     */
    void send_sequence(int stepPos);

    /**
     * @brief Set Stepper operation mode
     * @param v     @arg STEP       Single Step operation
     *              @arg HALFSTEP   Half Step operation
     */
    void set_operation_mode( OperationMode v );

    /**
     * @brief Rotate motor at specified angle with given speed
     * @param StMotorDirection Sets motor direction
     *        @arg CLOCKWISE
     *        @arg COUNTER_CLOCKWISE
     * @param Angle     Specify rotation angle
     * @param Speed     Specify Speed in Millisecond Per Step
     *                  50msps = 0.050 (It will take 50ms for 1 step)
     *
     * Calculations:
     * Move 360 degrees in 2 seconds
     *
     * operation mode = STEP                    operation mode = HALFSTEP
     *                           Time (in ms)
     *              speed = ------------------------
     *                       degrees/deg_per_step
     *
     *          2000    2000                            2000   2000
     *  speed =-------= ----= 10 = 0.010msps    speed =-------=----= 5 = 0.005msps
     *         360/1.8   200                           360/0.9  400
     *
     */
    void rotate_angle(RotMode StMotorDirection, int Angle, float Speed);

    /**
     * @brief Rotate motor with given steps with given time
     * @param StMotorDirection Sets motor direction
     *        @arg CLOCKWISE
     *        @arg COUNTER_CLOCKWISE
     * @param Steps     Specify Steps to rotate
     * @param Speed     Specify Speed in Millisecond Per Step
     *                  50msps = 0.050 (It will take 50ms for 1 step)
     *
     * Calculations:
     * Move 200 steps in 2 seconds
     *          Time (in ms)    2000
     *  speed = -------------= ------ = 10 = 0.010msps
     *             Steps         200
     *
     */
    void rotate_steps(RotMode StMotorDirection, int Steps, float Speed);

    /** Interface for syncronization mode setting
     *
     *  Example:
     *  @code
     *  StepperMotor    m( p21, p22, p23, p24 );
     *  int main() {
     *      m.set_sync_mode( StepperMotor::SYNCHRONOUS );
     *      ...
     *  @endcode
     *
     *  @param m motor rotate mode : ASYNCHRONOUS (default) or SYNCHRONOUS
     */
    void set_sync_mode( SyncMode m );

    /**
     *  @brief Check remaining distance that motor need to move
     *  software can check if the motor action completed in asynchronous mode
     *  @return remaining steps that motor need to go
     */
    int distance( void );

    /**
     *  @brief Pause/Resume the motor action
     *  @param sw use "true" for pause, "false" (default) for resume
     */
    void set_pause( int sw );

    /**
     * @brief Stop motor and reset values to default
     */
    void stop( void );

private:
    Ticker      t;
    BusOut      motor_out;

    static unsigned char  pattern_step_cw[ 4 ];  //  1 phase pulse pattern for motor control
    static unsigned char  pattern_step_acw[ 4 ];  //  1 phase pulse pattern for motor control
    static unsigned char  pattern_halfstep_cw[ 8 ];   //  1 phase pulse pattern for motor control
    static unsigned char  pattern_halfstep_acw[ 8 ];   //  1 phase pulse pattern for motor control
    unsigned char          *pattern;
    int                    pat_index_mask;
    OperationMode          phase_mode;
    RotMode     rot_mode;
    SyncMode    sync_mode;
    int         current_pos;
    int         target_pos;
    float       msps;
    float       max_msps;
    int         pause;

    void set_target_pos( int p );  //  target position setting interface
    void motor_maintain( void );   //  this function is called periodically by Ticker
};


#endif
