#include "BasicPacket.h"
#include "crc.h"
#include <string.h>

#ifdef _DEBUG
#include <cassert>
#endif

#define CRC_BASE_IDX 2

BasicPacket::BasicPacket()
{
    // init variables
    memset(_buffer, 0x0, sizeof(_buffer));
    _isVerified = false;
}

BasicPacket::BasicPacket(uint8_t *data)
{
    // the user takes the responsibility of making sure
    // they have sufficient memory for data buffer
    
    // copy data to internal buffer
    memcpy(_buffer, data, sizeof(_buffer));

    // set verify flag
    _isVerified = false;
}

BasicPacket::~BasicPacket()
{

}

void BasicPacket::generateCrc()
{
    _buffer[7] = crc8(_buffer, 3);

    // set verify flag
    _isVerified = true;
}

bool BasicPacket::verify()
{
    if (_isVerified)
    {
        return true;
    }
    else
    {
        return _buffer[7] == crc8(_buffer, 7);
    }
}

void BasicPacket::serialize(uint8_t *buffer)
{
    memcpy(buffer, _buffer, sizeof(_buffer));
}

void BasicPacket::setFid(uint8_t fid)
{
    // clear upper 4 bits
    _buffer[0] &= 0x0f;
    
    // set bits
    _buffer[0] |= fid << 4;

    // set verify flag
    _isVerified = false;
}

void BasicPacket::setSequenceID(uint8_t seqid)
{
    // clear lower 4 bits
    _buffer[0] &= 0xf0;

    // set bits
    _buffer[0] |= seqid & 0x0f;

    // set verify flag
    _isVerified = false;
}

void BasicPacket::setSourceID(uint8_t sid)
{
    // clear upper 4 bits
    _buffer[1] &= 0x0f;
    
    // set bits
    _buffer[1] |= sid << 4;

    // set verify flag
    _isVerified = false;
}

void BasicPacket::setDestinationID(uint8_t did)
{
    // clear lower 4 bits
    _buffer[1] &= 0xf0;
    
    // set bits
    _buffer[1] |= did & 0x0f;

    // set verify flag
    _isVerified = false;
}

void BasicPacket::setField(uint8_t idx, uint8_t data)
{
    // set payload based on index
    _buffer[CRC_BASE_IDX + idx] = data;

    // set verify flag
    _isVerified = false;
}

uint8_t BasicPacket::getFid()
{
    return (_buffer[0] & 0xf0) >> 4;
}

uint8_t BasicPacket::getSequenceID()
{
    return _buffer[0] & 0x0f;
}

uint8_t BasicPacket::getSourceID()
{
    return (_buffer[1] & 0xf0) >> 4;    
}

uint8_t BasicPacket::getDestinationID()
{
    return _buffer[1] & 0x0f;
}

uint8_t BasicPacket::getField(uint8_t idx)
{
    return _buffer[CRC_BASE_IDX + idx];
}

uint8_t BasicPacket::getCrc()
{
    if (_isVerified)
    {
        return _buffer[7];
    }
    else
    {
        return crc8(_buffer, 7);
    }
}

#ifdef _DEBUG
void BasicPacket::unit_test()
{
    // value test
    BasicPacket testObject;
    testObject.setFid(0x1);
    testObject.setField1(0x2);
    testObject.setField2(0x34);
    testObject.setField3(0x67);

    assert(0x1 == testObject.getFid());
    assert(0x2 == testObject.getField1());
    assert(0x34 == testObject.getField2());
    assert(0x67 == testObject.getField3());

    assert(false == testObject.verify());
    testObject.generateCrc();
    assert(true == testObject.verify());
    assert(0xeb == testObject.getCrc());

    // value test 2
    uint8_t testString[] = { 0x12, 0x34, 0x67, 0xeb };
    BasicPacket testObject2(testString);

    assert(0x1 == testObject2.getFid());
    assert(0x2 == testObject2.getField1());
    assert(0x34 == testObject2.getField2());
    assert(0x67 == testObject2.getField3());

    assert(true == testObject2.verify());
    testObject2.generateCrc();
    assert(true == testObject2.verify());
    assert(0xeb == testObject2.getCrc());
}
#endif