#ifndef ALOHATRANSCEIVER_H_
#define ALOHATRANSCEIVER_H_

#include "mbed.h"
#include "AlohaFrame.h"
#include "BasicPacket.h"
#include "RingBuffer.h"

// select suitable driver based on target platform
#ifdef DRIVER_SX1276
#include "sx1276-hal.h"
#endif

#ifdef DRIVER_INAIR
#include "sx1276-inAir.h"
#endif

// select FSK or LoRa
#define USE_MODEM_LORA  1
#define USE_MODEM_FSK   !USE_MODEM_LORA

#define RF_FREQUENCY                                    868000000 // Hz
#define TX_OUTPUT_POWER                                 14        // 14 dBm

#if USE_MODEM_LORA == 1

    #define LORA_BANDWIDTH                              0         // [0: 125 kHz,
                                                                  //  1: 250 kHz,
                                                                  //  2: 500 kHz,
                                                                  //  3: Reserved]
    #define LORA_SPREADING_FACTOR                       12         // [SF7..SF12]
    #define LORA_CODINGRATE                             1         // [1: 4/5,
                                                                  //  2: 4/6,
                                                                  //  3: 4/7,
                                                                  //  4: 4/8]
    #define LORA_PREAMBLE_LENGTH                        8         // Same for Tx and Rx
    #define LORA_SYMBOL_TIMEOUT                         5         // Symbols
    #define LORA_FIX_LENGTH_PAYLOAD_ON                  false
    #define LORA_FHSS_ENABLED                           false
    #define LORA_NB_SYMB_HOP                            4
    #define LORA_IQ_INVERSION_ON                        false
    #define LORA_CRC_ENABLED                            true

#elif USE_MODEM_FSK == 1

    #define FSK_FDEV                                    25000     // Hz
    #define FSK_DATARATE                                19200     // bps
    #define FSK_BANDWIDTH                               50000     // Hz
    #define FSK_AFC_BANDWIDTH                           83333     // Hz
    #define FSK_PREAMBLE_LENGTH                         5         // Same for Tx and Rx
    #define FSK_FIX_LENGTH_PAYLOAD_ON                   false
    #define FSK_CRC_ENABLED                             true

#else
    #error "Please define a modem in the compiler options."
#endif

#define RX_TIMEOUT_VALUE                                3500000   // in us
#define TX_TIMEOUT_VALUE                                2000000
#define BUFFER_SIZE                                     20        // Define the payload size here

// callback function for aloha frame
typedef void (*aloha_callback_func)(uint8_t *payload, uint8_t payload_length, uint8_t src_addr); 

class AlohaTransceiver
{
    // public type declaration
public:
    typedef struct
    {
        int8_t   Power;
        uint32_t Fdev;
        uint32_t Bandwidth;
        uint32_t BandwidthAfc;
        uint32_t Datarate;
        uint16_t PreambleLen;
        bool     FixLen;
        uint8_t  PayloadLen;
        bool     CrcOn;
        bool     IqInverted;
        bool     RxContinuous;
        uint32_t TxTimeout;
    } FskSettings_t;

    typedef struct
    {
        int8_t   Power;
        uint32_t Bandwidth;
        uint32_t Datarate;
        uint8_t  Coderate;
        uint16_t PreambleLen;
        uint16_t SymbolTimeout;
        bool     FixLen;
        uint8_t  PayloadLen;
        bool     CrcOn;
        bool     FreqHopOn;
        uint8_t  HopPeriod;
        bool     IqInverted;
        bool     RxContinuous;
        uint32_t TxTimeout;
    }LoRaSettings_t;
    // public member functions
public:
    AlohaTransceiver(uint8_t id=0);
    ~AlohaTransceiver();

    void boardInit();
    void updateSettings();
    void enable();
    void poll();

    // user callback function registration
    void registerType(AlohaFrame::AlohaType_t type, aloha_callback_func f);
    void deRegisterType(AlohaFrame::AlohaType_t type, aloha_callback_func f);
    
    // transmit data packet by user
    bool send(BasicPacket *packet);
    
    // read rssi and snr from previous transmission
    int16_t getRssi();
    int8_t getSnr();
    
    // read and write device id 
    uint8_t getDeviceID();
    void setDeviceID(uint8_t id);

#if USE_MODEM_LORA == 1
    LoRaSettings_t *getSettings();
#elif USE_MODEM_FSK == 1
    FskSettings_t *getSettings();
#else
    #error "Please define a modem in the compiler options."
#endif

// private member function
protected:
    // find next hop to reach the destination
    uint8_t findNextHop(uint8_t addr);
    
    // send acknowledgement back to the source
    void sendAck(AlohaFrame *inFrame);
    
    // send frame directly to the radio
    void sendFrame(AlohaFrame *frame);
    
    // flags
    bool getTxDoneFlag();
    void setTxDoneFlag();
    void clearTxDoneFlag();
    
    bool getAckedFlag(uint8_t addr);
    void setAckedFlag(uint8_t addr);
    void clearAckedFlag(uint8_t addr);


// private member properties
protected:
#if USE_MODEM_LORA == 1
    LoRaSettings_t Settings;
#elif USE_MODEM_FSK == 1
    FskSettings_t Settings;
#else
    #error "Please define a modem in the compiler options."
#endif
    
    // callback function table
    aloha_callback_func AlohaTypeCallbackTable[ALOHA_MAX_TYPE];

    // device id for 2nd layer protocol
    uint8_t deviceId;
    
    // the transceiver can handle 16 different destinations
    uint8_t seqid[16];
    
    // tx queue
    CircularBuffer<AlohaFrame *> AlohaTxQueue;
    
    // flags
    bool isTxDone;
    uint16_t isAcked;
    
    // CSMA/CA backoff timer
    Timer CSMABackoffTimer;
    
    // CSMA/CA state
    bool isBackoff;
    
    // CSMA/CA backoff time
    uint32_t CSMABackoffPeriod;
    
};
#endif
