#include "HuntEngine.h"
 //constructor
HuntEngine::HuntEngine()
{

}
//destructor that frees up memory when object goes out of scope
HuntEngine::~HuntEngine()
{

}
/* 
 * Function init:
 * Description: Initializes the game parameters
 *  Note:WIDTH is defined in LCD.h
 */
void HuntEngine::init(int predator_radius,int prey_size,int speed)
{
    _predator_radius = predator_radius;
    _prey_size = prey_size;
    _speed = speed;

    // x position on screen - 
    _px = GAP;
    

    // This code puts the game components (prey and the predator) in the middle
    _predator.init(_px,_predator_radius);
    _prey.init(_prey_size,_speed);
}
/* 
 * Function read_input:
 * Description: Obtains the direction and magnitude inputs from gamepad
 *  
 */
void HuntEngine::read_input(Gamepad &pad)
{
    _d = pad.get_direction();
    _mag = pad.get_mag();
}

/* 
 * Function draw:
 * Description: draw the elements in the LCD buffer:prey,predator and 
 *  
 */
void HuntEngine::draw(LCD &lcd)
{
    // draw the elements in the LCD buffer
    // pitch
    lcd.drawRect(0,0,WIDTH,HEIGHT,FILL_TRANSPARENT);
    lcd.drawLine(WIDTH/2,0,WIDTH/2,HEIGHT-1,2);
    //print the accumulating points
    print_points(lcd);
    // draw predators
    
    _predator.draw(lcd);
    // Prey being targeted by the predator
    _prey.draw(lcd);
}

void HuntEngine::update(Gamepad &pad)
{
    check_catch(pad);
    // important to update predator and prey before checking collisions so can
    // correct for it before updating the display
    _predator.update(_d,_mag);
    
    _prey.update();

    check_wall_collision(pad);
    check_predator_collisions(pad);
}

void HuntEngine::check_wall_collision(Gamepad &pad)
{
    // read current prey attributes
    POS2D prey_pos = _prey.get_pos();
    POS2D prey_velocity = _prey.get_velocity();

    // check if the prey hits top wall
    if (prey_pos.y <= 1) {  //  1 due to 1 pixel boundary
        prey_pos.y = 1;  // bounce off ceiling without going off screen
        prey_velocity.y = -prey_velocity.y;
        // audio feedback
        pad.tone(750.0,0.1);
    }
    // check if the prey hits bottom wall of the screen
    else if (prey_pos.y + _prey_size >= (HEIGHT-1) ) { // bottom pixel is 47
        // hit bottom
        prey_pos.y = (HEIGHT-1) - _prey_size;  // stops prey going off screen
        prey_velocity.y = -prey_velocity.y;
        // audio feedback
        pad.tone(750.0,0.1);
    }

    // update prey parameters
    _prey.set_velocity(prey_velocity);
    _prey.set_pos(prey_pos);
}

void HuntEngine::check_predator_collisions(Gamepad &pad)
{
    // read current prey attributes
    POS2D prey_pos = _prey.get_pos();
    POS2D prey_velocity = _prey.get_velocity();

    // check predator
    POS2D p_pos = _predator.get_pos();

    // see if predator has caught the prey by checking for Contact with the prey
    if (
        (prey_pos.y >= p_pos.y) && //top
        (prey_pos.y <= p_pos.y + _predator_radius) && //bottom
        (prey_pos.x >= _px) && //left
        (prey_pos.x <= _px + _predator_radius*2)  //right
    ) {
        // if it has, fix position and reflect x velocity
        prey_pos.x = _px + _predator_radius*2;
        prey_velocity.x = -prey_velocity.x;
        // audio feedback
        pad.tone(1000.0,0.1);
    }


    // write new attributes for the prey component
    _prey.set_velocity(prey_velocity);
    _prey.set_pos(prey_pos);
}

void HuntEngine::check_catch(Gamepad &pad)
{
    POS2D prey_pos = _prey.get_pos();
    POS2D pred_pos = _predator.get_pos();
    // Predator has garnered
   
   if (prey_pos.x==pred_pos.x && prey_pos.y==pred_pos.y) {
        _predator.add_points();
        _prey.init(_prey_size,_speed);
        pad.tone(1500.0,0.5);
        pad.leds_on();
        wait(0.5);
        pad.leds_off();
    }
    
}

/* 
 * Function print_points:
 * Description: The function fetches points for predator one and two and prints the on LCD screen
 *  Note:WIDTH is defined in LCD.h
 */
void HuntEngine::print_points(LCD &lcd)
{
    // get points from the predator
    int p_point = _predator.get_points();
    // print the respective points garnered to LCD screen
    char predascore[10];
    sprintf(predascore,"%2d",p_point);
    lcd.printString(predascore,2,1);  // print the resuilts at the top left section
    
}