/** Inventory Class
 * @brief Library for maintaining an inventory
 * @author George Sykes [el18gs]
 * @date 11 May 2020
 * @version 1.1
 */

#ifndef GAMEENGINE_H
#define GAMEENGINE_H

/** Game enginer library that provides all the functios necessary to run the game
 * @author George Sykes [el18gs]
 * @date 13 May 2020
 * @version 1.0
 */

#include "mbed.h"
#include "Gamepad.h"
#include "N5110.h"
#include "SDFileSystem.h"
#include "FX0S8700CQ.h"
#include "Ghost.h"
#include "Inventory.h"
#include "tooling.h"
#include <vector>
#include <string>

/** struct for FSM used to represent menus throughout the game */
struct State {
    int output;  //**< Output value of the FSM */
    int nextState[2];  //**< The next state to transition too {up, down} */
};

// Constant FSM's
// main Menu FSM
const State mainMenuFsm[4] = {
    //   line next{up, down}
    {0,{2,1}}, // State: 0
    {2,{0,2}}, // State: 1
    {4,{1,0}}, // State: 2
};

// Settings menu
const State settingsFsm[2] = {
    //   line next{up, down}
    {2,{1,1}}, // State: 0
    {4,{0,0}}, // State: 1
};

/** gameEngine class provides an interface for running the game
* @author George Sykes [el18gs]
* @date 13 May 2020
* @veersion 1.0
*/
class gameEngine
{
public:
    // Cnstructor
    /** Create a game engine object and initilise the periphials
    * @param sd pointer to the sd card object
    * @param pad pointer to the gamepad object
    * @param lcd pointer to the LCD object
    */
    gameEngine(SDFileSystem &sd, 
                Gamepad &pad, 
                N5110 &lcd);
                
    /** Display the welcome screen allowing the user to adjust the contrast, continue
    * by pressing A
    * @param sd pointer to the sd card object
    * @param lcd pointer to the LCD object
    * @param pad pointer to the gamepad object
    * @param g_buttonA_flag pointer to the global variable that will be set to 1 if the A button has been pressed
    */
    void welcome(SDFileSystem &sd, 
                N5110 &lcd, 
                Gamepad &pad, 
                volatile int &g_buttonA_flag);
    
    /** Display the game menu, there are three options; catch ghosts, inventory, settings
    * @param lcd pointer to the LCD object
    * @param g_buttonA_flag pointer to the global variable that will be set to 1 if the A button has been pressed
    * @param g_buttonB_flag pointer to the global variable that will be set to 1 if the B button has been pressed
    * @param g_buttonX_flag pointer to the global variable that will be set to 1 if the X button has been pressed
    * @return integer relating to which option was selected 'catch ghosts' = 0, 'inventory' = 1, 'settings' = 2
    * @note use X and B to move the selection up and down, press A to select
    */
    int game_menu(N5110 &lcd, 
                volatile int &g_buttonA_flag, 
                volatile int &g_buttonB_flag, 
                volatile int &g_buttonX_flag);
    
    /** Provide the user with two challenges if they complete both they will get an extra ghost.
    * The first challenge is rotating the gamepad to the right direction. The seccond is moving a crosshair onto a randomly moving ghost.
    * @note in the seccond the ghost is intentionally jittering and moving quickly as it looks more ghost like
    * @param inventory pointer to the users inventory
    * @param accel pointer to the accelerometer object
    * @param sd pointer to the sd card object
    * @param lcd pointer to the LCD object
    * @param pad pointer to the gamepad object
    * @param g_buttonA_flag pointer to the global variable that will be set to 1
    * if the A button has been pressed
    * @param left_LED pointer to the bus out object representing the left LED's
    * @param right_LED pointer to the bus out object representing the right LED's
    */
    void catch_ghosts(Inventory &inventory, 
                FX0S8700CQ &accel, 
                SDFileSystem &sd, 
                N5110 &lcd, 
                Gamepad &pad, 
                volatile int &g_buttonA_flag, 
                BusOut &left_LED, 
                BusOut &right_LED);
    
    /** Settings menu alows the user to change the contrast and button delay time 
    * (how long the game waits after a button press)
    * @note contrast is limited to 0.4 to 0..6
    * @note use X and B to move up and down respectivly and A to select the option
    * @todo change button sensitivity to background timer that won't allow the 
    * button to be set again until a period of time has passed (note other functions
    * may happen in the meantime)
    * @todo add a method of saving the settings for future use
    * @param lcd pointer to the LCD object
    * @param pad pointer to the gamepad object
    * @param g_buttonA_flag pointer to the global variable that will be set to 1 if the A button has been pressed
    * @param g_buttonX_flag pointer to the global variable that will be set to 1 if the X button has been pressed
    * @param g_buttonB_flag pointer to the global variable that will be set to 1 if the B button has been pressed
    * @param g_buttonStart_flag pointer to the global variable that will be set to 1 if the Start button has been pressed
    * @param g_buttonTesting to the global variable that tells button X to use g_buttonSensitivityTest as its wait
    * time not g_buttonSensitivity
    * @param g_buttonSensitivityTest to the global variable that acts as a tempory waiting time the X button during testing
    * this is to allow the user to test the setting before saving it
    * @param g_buttonSensitivity pointer to the global variable that tells the buttons how long to wait after being pressed
    */
    void settings(N5110 &lcd, 
                Gamepad &pad, 
                volatile int &g_buttonA_flag, 
                volatile int &g_buttonX_flag, 
                volatile int &g_buttonB_flag,
                volatile int &g_buttonStart_flag,
                volatile bool &g_buttonTesting,
                volatile int &g_buttonSensitivityTest,
                volatile int &g_buttonSensitivity);

private:
    void adjustContrast(N5110 &lcd, 
                Gamepad &pad, 
                volatile int &g_buttonA_flag);
    
    void buttonDelay(N5110 &lcd,
                     Gamepad &pad,
                     volatile int &g_buttonA_flag,
                     volatile int &g_buttonX_flag,
                     volatile bool &g_buttonTesting,
                     volatile int &g_buttonSensitivityTest,
                     volatile int &g_buttonSensitivity);
    
    void ghostCatchTrial(FX0S8700CQ &accel, 
                    N5110 &lcd, 
                    SDFileSystem &sd, 
                    Gamepad &pad, 
                    BusOut &left_LED, 
                    BusOut &right_LED, 
                    volatile int &g_buttonA_flag);
    
    void angleDetectionTechnicalSub(FX0S8700CQ &accel, 
                    BusOut &left_LED, 
                    BusOut &right_LED);
    
    bool drawCrosshairs(Vector2D ideal, 
                    int** ghost, 
                    N5110 &lcd, 
                    Gamepad &pad, 
                    volatile int g_buttonA_flag);
    
    void randomMove(Vector2D &ideal);
    
    bool ghostHit( int xGhost, 
                    int yGhost, 
                    int xJoy, 
                    int yJoy);
};

#endif