/** @file Inventory.h
*   @brief Inventory library containing the tools used to generate and manage
*   @brief the games inventory
*/
#ifndef INVEN_H
#define INVEN_H

#include "SDFileSystem.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Ghost.h"
#include "tooling.h"
#include <string>
#include <vector>
#include <iostream>

/** @pulic vector of Ghosts used to hold the inventory data
*/
typedef std::vector<Ghost> ghostvec;


/** FSM for inventory struct */
struct inven_state {
    int uid; /**< UID of the ghost */
    int next[2]; /**< next{up, down} */
    std::string type; /**< Type of ghost (stored as string not enum)*/
    std::string name; /**< Name of ghost */
    int attack; /**< Attack value of ghost */
    int defense; /**< Defense value of ghost*/
    int level; /**< Level of the ghost */
    int xp; /**< How much XP the ghost has */
    int value; /**< The ghosts value */
    int hp_max; /**< maximum HP of the ghost */
    int hp; /**< current HP value of the ghost */
};

/** Inventory Class
 * @brief Library for maintaining an inventory
 * @author George Sykes [el18gs]
 * @date 11 May 2020
 * @version 1.1
 */
class Inventory
{
public:
    // Constructor
    /** Create a Inventory object by importing all the .ghost files in the
    * @brief /ghosts directory
    * @param sd Pointer to an SDFileSystem object
    */
    Inventory(SDFileSystem &sd);

    /** Regenerate the inventory after a change has been made
    * @param sd Pointer to an SDFileSystem object
    */
    void regen(SDFileSystem &sd);

    /** List all the UID's of the ghosts in the inventory
    * @param path The name of the .ghost file to import
    * @return A vector of integers
    */
    std::vector<int> list_ghost_uids(void);

    /** returns a copy of a ghost in the inventory defined by its UID
    * @param uid defines the UID of the ghost to import
    * @return a Ghost object
    */
    Ghost get_ghost_by_uid(int uid);

    /** Sell one of the ghosts in the inventory
    * @param uid defines the UID of the ghost to sell
    * @param sd Pointer to an SDFileSystem object
    */
    void sell_ghost_by_uid(int uid, SDFileSystem &sd);
    
    /** Feed one of the ghosts in the inventory
    * @param uid defines the UID of the ghost to sell
    * @param sd Pointer to an SDFileSystem object
    */
    void feed_ghost_by_uid(int uid, SDFileSystem &sd);

    /** Display GUI of the inventory, allowing the user to; view ghosts in the
    * inventory and sell them.
    * @param sd Pointer to an SDFileSystem object
    * @param lcd pointer to an N5110 object for displaying the GUI onto
    * @param pad pointer to an Gamepad object for controlling the GUI
    * @param X_flag pointer to the global variable that the program sets to 1 when the X button is pressed.
    * @param Start_flag pointer to the global variable that the program sets to 1 when the Start button is pressed.
    * @param A_flag pointer to the global variable that the program sets to 1 when the A button is pressed.
    * @note the global flags must be modified by ISR as the function does not check the buttons itself.
    */
    void display_inventory( SDFileSystem &sd,
                            N5110 &lcd,
                            Gamepad &pad,
                            volatile int &g_buttonX_flag,
                            volatile int &g_buttonStart_flag,
                            volatile int &g_buttonA_flag);

private:
    // Functions
    stringvec list_ghosts(std::string, SDFileSystem &sd);
    void print_coin(SDFileSystem &sd, N5110 &lcd);
    std::vector<inven_state> gen_ghost_fsm();

    // Variables
    ghostvec _ghosts;
    int _gold;
    int _food_low;
    int _food_med;
    int _food_high;

    static const string _root;
};

#endif