/** @file Ghost.h
*   @brief Ghosts library contains the necessary class, enums and typedefs
*   @brief to genearate and manipulate .ghost files for the game
*/ 

// Ensure that the library is not importated twice
#ifndef GHOST_H
#define GHOST_H

// Import necessay libraries
#include "SDFileSystem.h"
#include "tooling.h"
#include <string>
#include <vector>
#include <iostream>

/** @public Enumerated Type called Type Used to illustrate the type or species 
* of a ghost object, chances of each ghost being caught displayed next to type
*/
enum Type {BASIC, /**< 50%*/
            ECTO, /**< 20% */
            POLTER, /**< 15% */
            ORB, /**< 10% */
            FUNNEL /**< 5% */
            };

/** Ghost Class
 * @version 1.0
 * @brief Library for generating and saving .ghost files for the game. Some of
 * @brief the files need to be imported from the SD others are generated.
 * @author George Sykes [el18gs]
 * @date 11 May 2020

@code
sample.ghost file
1.ghost:
/Type
Basic
/Name
Casper
/Attack
10
/Defense
10
/Level
1
/XP
0
/Value
10
/HP_max
10
/HP
10
/UID
1
@endcode
*/
class Ghost
{
public:
    // Constructors
    /** Create a Ghost object by importing a .ghost file
    * @param path The name of the .ghost file to import
    * @param root The directory to look for the file in
    * @param sd Pointer to an SDFileSystem object
    */
    Ghost(const std::string path, const std::string root, SDFileSystem &sd);
    
    /** Create a Ghost object using two numbers to define its type and name
    * @param type an interger number between 0 and 100 to define the type of ghost
    * @param nameNumber an integer number between 0 and 20 to define the name of the ghost
    * @param root the directory to save the Ghost object in
    * @param sd Pointer to an SDFileSystem object
    * @note this constructor will save the object as a .ghost file using the @ref save(void)
    * @note the UID (Unique ID) of this object will be generated by @ref gen_uid(void)
    */
    Ghost(int type, int nameNumber, const std::string root, SDFileSystem &sd);
    
    /** List all files in the directory path passed
    * @param Path directory to list
    * @param sd Pointer to an SDFileSystem object
    */
    void listdir(std::string path, SDFileSystem &sd);
    
    /** Save the current ghost
    * @param sd Pointer to an SDFileSystem object
    * @note save the current ghost as a .ghost file, the name of the file is the UID
    */
    void save(SDFileSystem &sd);
    
    /** 'Sell' the ghost, in reality this means deleting its file and returning the value of the ghost
    * @param sd Pointer to an SDFileSystem object
    * @return an integer
    */
    int sell(SDFileSystem &sd);
    
    /** 'feed' the ghost, this means give it XP equal to the ammount passed
    * @param ammount of food to feed, this is equal to the xp increase
    * @param sd Pointer to an SDFileSystem object
    * @note after this function the ghost will be saved using @ref save()
    */
    void feed(int ammount, SDFileSystem &sd);
    
    /** Print all the member variables over serial
    * @note primarily used in debugging & development, can be removed if more space needed in final version
    */
    void print_all(void);
    
    /** Get the type of ghost as an enum
    * @notes uses the type stored in _type
    * @return an enumerated class value
    */
    Type get_type_enum(void);
    
    /** Get a string version of the type of ghost
    * @notes uses the type stored in _type
    * @return an c++ string
    */
    std::string get_type_string(void);
    
    /** Get the name of the ghost
    * @notes uses the name stored in _name
    * @return an c++ string
    */
    std::string get_name(void);
    
    /** Get the attack value of the ghost
    * @notes uses the attack value stored in _attack
    * @return an interger
    */
    int get_attack(void);
    
    /** Get the defense value of the ghost
    * @notes uses the defense value stored in _defense
    * @return an integer
    */
    int get_defense(void);
    
    /** Get the level of the ghost
    * @notes uses the level stored in _level
    * @return an integer
    */
    int get_level(void);
    
    /** Get the ammount of XP the ghost has
    * @notes uses the XP ammount stored in _xp
    * @return an integer
    */
    int get_xp(void);
    
    /** Get the value of the ghost
    * @notes uses the value stored in _value
    * @return an integer
    */
    int get_value(void);
    
    /** Get the maximum HP of the ghost
    * @notes uses the maximum HP stored in _hp_max
    * @return an integer
    */
    int get_hp_max(void);
    
    /** Get the current hp value of the ghost
    * @notes uses the hp value stored in _hp
    * @return an integer
    */
    int get_hp(void);
    
    /** Get the UID of the ghost
    * @notes uses the UID (Unique ID) stored in _uid
    * @return an integer
    */
    int get_uid(void);

private:
    // Methods
    int gen_uid(SDFileSystem &sd);
    bool hasEnding (std::string const &fullString, std::string const &ending);

    std::string type_to_string(Type type);
    Type string_to_type(std::string type);
    void delete_file(const char filename[], SDFileSystem &sd);

    Type _type;
    std::string _name;
    int _attack;
    int _defense;
    int _level;
    int _xp;
    int _value;
    int _hp_max;
    int _hp;
    int _uid;
    static const string _suffix;
    std::string _root;

    stringvec _names;
};

#endif