/** @file FX0S8700CQ.h
*   @brief Library to interface with the FX0S8700CQ magnetometer
*/ 

// Protect library from being imported twice
#ifndef FX0S8700CQ_H
#define FX0S8700CQ_H

#include "mbed.h"
#include <math.h>

// Define symbols used throughout class
#define FAST 1
#define NORMAL 0
#define WRITE 0

/** Struct to handle data */
struct  Data 
{
    float ax;       /**< Acceleration in the X axis */
    float ay;       /**< Acceleration in the Y axis */
    float az;       /**< Acceleration in the Z axis */
    
    float mx;       /**< Magnetisim in the X axis */
    float my;       /**< Magnetisim in the X axis */
    float mz;       /**< Magnetisim in the X axis */
    
    float roll;     /**< Roll in radians*/
    float pitch;    /**< Pitch in radians */
    float yaw;      /**< Yaw in radians */
};


/** FX0S8700CQ Class
 * @brief Class used to represent the interface
 * @author George Sykes [el18gs]
 * @date 12 May 2020
 * @version 1.1
 */
class FX0S8700CQ
{
    /** This class manages one FX08700CQ sensor
    * @TODO Calibration sequences
    * @TODO Temperature sensor
    * @TODO Drop detection
    * @TODO Pulse detection
    * @TODO Sleep mode
    * @TODO Wake on pickup
    */
public:
    /** This is the default constructor and sets up the I2C interface
    * @param address this defines what adress the sensor should has, default value 0x1D
    * @param speed The I2C standard speed to use, options are FAST or NORMAL default is FAST
    * @param auto_update whether the gather data program should be automatically run at intervals default is false
    * @param period how frequently should the data be automatically updated in ms, default 10 ms
    * @param magn_info whether to collect magnitude info, defualt true
    * @param SDA_pin what pin will is the Serial Data pin, defualts to I2C_SDA
    * @param SCL_pin what pin will is the clock pin, defualts to I2C_SCL
    */
    FX0S8700CQ(char address = 0x1D,
               int speed = FAST,
               bool auto_update = false,
               int period = 10,
               bool mag_info = true,
               PinName sda = I2C_SDA,
               PinName scl = I2C_SCL);

    /** Destructor function to call once the object falls out of scope */
    ~FX0S8700CQ();
    
    // This function is used to write values to the sensors register
    /** Write a value to an adress in the chips register
    * @param address what register to write too
    * @param value what value to write into the register
    * @return a bool, true if successful else false
    */
    bool write_register(char address, char value);

    // This function is used to read values from the sensors register
    /** Read from an address in chips registers for an number of bytes
    * @param reg the address of the register to read from
    * @param num_bytes how many bytes to read out of memory
    * @param bytes where to store the data after its been recieved
    */
    void read_register(char reg, int num_bytes, char bytes[]);

    /** This function reads Data from the sensor and stores it */
    void read_data();

    /** @var Data sensor
    * Stores the sensor data in a struct
    */
    Data sensor;

private:
    char _address;

    // Ticker interupt object to retrieve new sensor Data
    Ticker _tick_read;
    void _tick_service();

    // I2C interface object
    I2C* _i2c;
    
    void accel_calibration();

};

#endif