/*
    FX0S8700CQ (3 Axis accelerometer/magnetometer sensor) library
    Written By George Sykes
    REV 1.0
*/
// Import the header file
#include "FX0S8700CQ.h"

// FX0S8700CQ internal register addresses
#define FX0S8700CQ_STATUS         0x00
#define FX0S8700CQ_WHOAMI         0x0D
#define FX0S8700CQ_XYZ_DATA_CFG   0x0E
#define FX0S8700CQ_CTRL_REG1      0x2A
#define FX0S8700CQ_M_CTRL_REG1    0x5B
#define FX0S8700CQ_M_CTRL_REG2    0x5C
#define FX0S8700CQ_WHOAMI_VAL     0xC7
#define FX0S8700CQ_OUT_X_MSB_REG  0x01
#define FX0S8700CQ_OFF_X_REG      0x2F
#define FX0S8700CQ_OFF_Y_REG      0x30
#define FX0S8700CQ_OFF_Z_REG      0x31
#define RAD2DEG                   57.2957795131f

FX0S8700CQ::FX0S8700CQ(char address,
                       int speed,
                       bool auto_update,
                       int period,
                       bool mag_info,
                       PinName sda,
                       PinName scl)
{

    _i2c = new I2C(sda,scl);        // Construct a new i2c interface objecct

    // Set the address variable
    _address = (address << 1);

    // Set the speed of the i2c interface depending on wether fast or normal
    // mode was passed
    if(speed == FAST) {
        _i2c->frequency(400000); // Set speed to 400KHz
    } else if (speed == NORMAL) {
        _i2c->frequency(100000); // Set speed to 100KHz
    } else {
        // If the speed setting is invalid print an error message
        error("Incorrect i2c speed designation, passed value was: %i", speed);
    }
    char chip_check[1];
    read_register(FX0S8700CQ_WHOAMI, 1, chip_check);
    // Use the * to access the data in the char pointer
    if (*chip_check != 0xC7) { // if correct ID not found, hang and flash error message
        error("Incorrect ID!");
    } else {
        printf("CHIP FOUND");
    }

    //if(auto_update) {
    //    _tick_read.attach(this, &FX0S8700CQ::read_data, period/1000);
    //    printf("Ticker set up complete\n");
    //}

    char data;

    // write 0000 0000 = 0x00 to accelerometer control register 1 to place
    // FX0S8700CQ into standby
    // [7-1] = 0000 000
    // [0]: active=0
    data = 0x00;
    write_register(FX0S8700CQ_CTRL_REG1, data);

    // write 0001 1111 = 0x1F to magnetometer control register 1
    // [7]: m_acal=0: auto calibration disabled
    // [6]: m_rst=0: no one-shot magnetic reset
    // [5]: m_ost=0: no one-shot magnetic measurement
    // [4-2]: m_os=111=7: 8x oversampling (for 200Hz) to reduce magnetometer noise
    // [1-0]: m_hms=11=3: select hybrid mode with accel and magnetometer active
    data = 0x1F;
    write_register(FX0S8700CQ_M_CTRL_REG1, data);

    // write 0010 0000 = 0x20 to magnetometer control register 2
    // [7]: reserved
    // [6]: reserved
    // [5]: hyb_autoinc_mode=1 to map the magnetometer registers to follow
    // the accelerometer registers
    // [4]: m_maxmin_dis=0 to retain default min/max latching even though not used
    // [3]: m_maxmin_dis_ths=0
    // [2]: m_maxmin_rst=0
    // [1-0]: m_rst_cnt=00 to enable magnetic reset each cycle
    data = 0x20;
    write_register(FX0S8700CQ_M_CTRL_REG2, data);

    // write 0000 0001= 0x01 to XYZ_DATA_CFG register
    // [7]: reserved
    // [6]: reserved
    // [5]: reserved
    // [4]: hpf_out=0
    // [3]: reserved
    // [2]: reserved
    // [1-0]: fs=01 for accelerometer range of +/-4g range with 0.488mg/LSB
    data = 0x01;
    write_register(FX0S8700CQ_XYZ_DATA_CFG, data);

    // write 0000 1101 = 0x0D to accelerometer control register 1
    // [7-6]: aslp_rate=00
    // [5-3]: dr=001 for 200Hz data rate (when in hybrid mode)
    // [2]: lnoise=1 for low noise mode
    // [1]: f_read=0 for normal 16 bit reads
    // [0]: active=1 to take the part out of standby and enable sampling
    data = 0x0D;
    write_register(FX0S8700CQ_CTRL_REG1, data);

    accel_calibration();

}

// destructor is called when the object goes out of scope
FX0S8700CQ::~FX0S8700CQ()
{
    delete _i2c;        // free memory
}


bool FX0S8700CQ::write_register(char address, char value)
{
    char data[2];
    data[0] = address;
    data[1] = value;

    // i2c->write(chip address, data to send, number of bytes to send, repeat);
    int nack = _i2c->write(_address, data, 2);
    if(nack == 0) {
        return true;
    } else {
        return false;
    }
}

void FX0S8700CQ::read_register(char reg, int num_bytes, char bytes[])
{
    // i2c->write(chip address, data to send, number of bytes to send, repeat);
    int nack = _i2c->write(_address, &reg, 1, true);
    if(nack) {
        error("No acknowledgement received!");  // if we don't receive acknowledgement, send error message
    }

    // i2c->read(chip address, result location, number of bytes to read);
    nack = _i2c->read(_address,bytes,num_bytes);  // read bytes
    if (nack)
        error("No acknowledgement received!");  // if we don't receive acknowledgement, send error message
}

void FX0S8700CQ::read_data()
{
    char data[13];
    read_register(FX0S8700CQ_STATUS, 13, data);

    // copy the 14 bit accelerometer byte data into 16 bit words
    int acc_x = (int16_t)(((data[1] << 8) | data[2]))>> 2;
    int acc_y = (int16_t)(((data[3] << 8) | data[4]))>> 2;
    int acc_z = (int16_t)(((data[5] << 8) | data[6]))>> 2;

    // copy the magnetometer byte data into 16 bit words
    int mag_x = (int16_t) (data[7] << 8) | data[8];
    int mag_y = (int16_t) (data[9] << 8) | data[10];
    int mag_z = (int16_t) (data[11] << 8) | data[12];

    // 0.488 mg/LSB in 4 g mode (8.1 data sheet)
    sensor.ax = 0.488e-3*acc_x;
    sensor.ay = 0.488e-3*acc_y;
    sensor.az = 0.488e-3*acc_z;

    // the magnetometer sensitivity is fixed at 0.1 μT/LSB
    sensor.mx = 0.1e-6*mag_x;
    sensor.my = 0.1e-6*mag_y;
    sensor.mz = 0.1e-6*mag_z;

    sensor.roll = atan2f(sensor.ay, sensor.az);
    sensor.roll =  sensor.roll * RAD2DEG;

    float ay_squared = pow(sensor.ay, 2);
    float az_squared = pow(sensor.az, 2);

    float squared = sqrt(ay_squared + az_squared);
    sensor.pitch = atanf(sensor.ax/squared);
    sensor.pitch = sensor.pitch * RAD2DEG;
}


void FX0S8700CQ::accel_calibration()
{

    char X_Accel_offset, Y_Accel_offset, Z_Accel_offset;

    read_data();

    X_Accel_offset = sensor.ax/ 4 * (-1);         // Compute X-axis offset correction value

    Y_Accel_offset = sensor.ay / 4 * (-1);         // Compute Y-axis offset correction value

    Z_Accel_offset = (4096 - sensor.az) / 4 * (-1);          // Compute Z-axis offset correction value



    write_register(FX0S8700CQ_OFF_X_REG, X_Accel_offset);

    write_register(FX0S8700CQ_OFF_Y_REG, Y_Accel_offset);

    write_register(FX0S8700CQ_OFF_Z_REG, Z_Accel_offset);

    write_register(FX0S8700CQ_CTRL_REG1, 0x35);          // Active mode again

}