#ifndef GAME_H
#define GAME_H

#include <cmath>
#include <cstdlib>
#include "mbed.h"
#include "Gamepad.h"
#include "N5110.h"
#include "Ball.h"
#include "Splash.h"
#include "SDFileSystem.h"
#include "SDCRC.h"
#include "GoalSounds.h"

/** Game class
@brief Class that controls how the game flows/functions.
@version 1.0
@author Kaif Kutchwala
@date May 2020
*/

class Game {
    private:
        //objects
        N5110 *_lcd;
        Gamepad *_pad;
        Ball *_ball;

    public:
        Game(N5110 &lcd, Gamepad &pad, Ball &ball);
        ~Game();
        /** @brief initialise variables, checks for initial highscore*/
        void init();
        /** @brief combines all other functions to form main gameplay loop)*/
        void play();
        /** @brief reads values user inputs i.e. direction and power (x and y)*/
        void readInput();
        /** @brief updates score on screen*/
        void updateScore();
        /** @brief updates number of lives on leds */
        void updateLives();
        /** @brief updates speed based on score */
        void updateSpeed();
        /** @brief updates speed based on score */
        void updateLevel();
        /** @brief updates all leds based on given value
        *   @param val @details array containing values to write to each led
        */
        void updateLeds(int val[6]);
        /** @brief plays sound effect for when new highscore is reached
        * @param sound @details determines which sound to play (1-miss,2-goal,
        * 3-game over, 4-extra life)
        */
        void playGoalSound(int sound);
    
        //accessors and mutators
        /** @brief sets highscore 
        * @param score @details integer value to be set as highscore
        */
        void set_highscore(int score);
        
        /** Gets highscore value
        * @returns a positive integer value
        */
        int get_highscore();

        /** Gets score value
        * @returns a positive integer value that is the score of the player
        */
        int get_score();

    private:
        //functions
        /** @brief takes pointer input for aim on x-axis */
        void pointer_input();

        /** @brief takes power input for aim on y-axis */
        void power_meter_input();
    
        /** @brief converts _x_val from pointer range (8-72) to screen range (0-84) */
        void convert_to_shot_x();
    
        /** @brief converts _y_val from power meter range (0-20) to goal range (0-24) */
        void convert_to_shot_y();
    
        /** @brief generates random value for given limit 
         * @param limit @details sets max limit for random generator
         * @returns random number in range 1-limit
         */
        int random_level_gen(int limit);
        /** @brief Prints "Goal" or "Miss" on screen based on input integer
         *  @param n @details tells function what message to print 
         *  (0- Miss 1- Goal)
         */
        void print_goal_message(int n);
    
        //variables
        bool _is_goal;
        int _score;
        int _highscore;
        int _lives;
        int _new_lives_threshold;
        int _level;
        float _x_val;
        float _y_val;
        int _shot_x;
        int _shot_y;
        float _speed;
        float _new_speed_threshold;
};
#endif