#ifndef PUSHINGENGINE_H
#define PUSHINGENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Ppl.h"
#include "Box.h"
#include "Cross.h"
#include "Menu.h"

/** PushingEngine Class
@author Zhenwen liao, University of Leeds
@brief Controls the Pushing and Determining Move in the Sokoban game
@date 04 2019
*/
class PushingEngine
{

public:
    /** Constructor */
    PushingEngine();

    /** Destructor */
    ~PushingEngine();

    /** Initialise the first position of two boxes, ppl, two crosses and initial reading of Menu parameter
    * @param box1_x @details the first x position of box1 (int)
    * @param box1_y @details the first y position of box1 (int)
    * @param box2_x @details the first x position of box2 (int)
    * @param box2_y @details the first y position of box2 (int)
    * @param ppl_x @details the first x position of ppl (int)
    * @param ppl_y @details the first y position of ppl (int)
    * @param cross1_x @details the first x position of cross1 (int)
    * @param cross1_y @details the first y position of cross1 (int)
    * @param cross2_x @details the first x position of cross2 (int)
    * @param cross2_y @details the first y position of cross2 (int)
    * @param menu_x @details the initial reading of level for menu (int)
    * @param menu_y @details the initial reading of level confirmation for menu (int)
    */
    void init(int box1_x,int box1_y,int box2_x,int box2_y,int ppl_x,int ppl_y,
              int cross1_x,int cross1_y,int cross2_x,int cross2_y,int menu_x,int menu_y);

    /** Read the inputs from Gamepad
    * @details Read the potentialmeter, botton A,B,X,Y
    *          the reading of potentialmeter is used to control the tone of A,B,X,Y
    *          being pressed
    */
    void read_input(Gamepad &pad);

    /** Update the Pushing and determining proccess
    * @details First,check box and ppl touching or not
    *          Second,check if next move of ppl will kick the box out off screen
    *          Third, update the parameter accorading to the determination of above function
    *          Fourth, reset the indicator for function
    *          Fifth, check if ppl will cover the box after next move
    *          Sixth, check if box and cross match each other and set the scores
    */
    void update(Gamepad &pad,int barrier_x,int barrier_y);

    /** Draw all the oject on the screen*/
    void draw(N5110 &lcd,int barrier_x,int barrier_y);

    /** Set the score */
    int set_score();

    /** Get the Menu
    * @return the parameter of current Menu state
    */
    Vector2D get_menu(N5110 &lcd,Gamepad &pad);

    /** Reset the menu parameter to level 1 and not confrim*/
    void reset_menu();

    /** Print a set of short introuduction of the game */
    void introduction(N5110 &lcd,Gamepad &pad);
    
    /**
     * @brief Check if ppl is touching box1
     * @details First get the current position of ppl and box1.
     *          Then, calculate the x,y position difference between
     *          box1 and ppl respectively.
     *          If y positions are the same,x of ppl is 8-pixel bigger than x of box1
     *          that is to say ppl is on the right hand side of the box1 and box can be
     *          pushed to left. In this case, the indicactor for box1 _s will be set to 1.
     *          And so as the rest of cases,indicactor will be set to 2 for pushing right,
     *          3 for pusing down, 4 for pushing up and 0 for ppl and box1 not touching.
     */
    void check_ppl_box1_touching(Gamepad &pad);
    
    /**
     * @brief Check if ppl is touching box2, Basically the same as check_ppl_box1_touching
     * @details If y positions are the same,x of ppl is 8-pixel bigger than x of box2
     *          that is to say ppl is on the right hand side of the box2 and box can be
     *          pushed to left. In this case, the indicactor for box1 _r will be set to 1.
     *          And so as the rest of cases,indicactor will be set to 2 for pushing right,
     *          3 for pusing down, 4 for pushing up and 0 for ppl and box1 not touching.
     */
    void check_ppl_box2_touching(Gamepad &pad);
    
    /**
     * @brief Hold the position of Ppl when box1 face against the wall
     * @details There is a scenario which the box is against the wall, if ppl try to push
     *          the box to wall side. The box will go off screen, and ppl will take its position.
     *          To fix this, i implement this function which works with the check_ppl_box1_touching function.
     *          First, check the position of the box,to see if it is on the egde of the screen.
     *          Then, use the indicator from box&ppl touching function to see if the off-screen will happen.
     *          If the x position of box1 is equal to 2 which means it cant be move left any more.
     *          Then, we check if the touching indicator _s is equal to 1. If true, we will
     *          set a new indicator _temp to 1. This indicator _temp is then use to freeze the ppl posision
     *          in its own class fucntion. The rest of the cases is implemented in the same way.
     */
    void hold_ppl_box1_wall(Gamepad &pad);
    
    /**
     * @brief Hold the position of Ppl when box2 face against the wall. Basically the same as hold_ppl_box1_wall
     * @details There is a scenario which the box is against the wall, if ppl try to push
     *          the box to wall side. The box will go off screen, and ppl will take its position.
     *          To fix this, i implement this function which works with the check_ppl_box2_touching function.
     *          First, check the position of the box,to see if it is on the egde of the screen.
     *          Then, use the indicator from box&ppl touching function to see if the off-screen will happen.
     *          If the x position of box1 is equal to 2 which means it cant be move left any more.
     *          Then, we check if the touching indicator _r is equal to 1. If true, we will
     *          set a new indicator _temp to 1. This indicator _temp is then use to freeze the ppl posision
     *          in its own class fucntion. The rest of the cases is implemented in the same way.
     */
    void hold_ppl_box2_wall(Gamepad &pad);
    
    /** Check if ppl cover the box
    * @details Check if move ppl cover the box then result input back into the update function to determine
    *          if ppl will be moved or stay at it last position
    * @return  True if moving ppl will cover box; false if not         
    */
    bool ppl_cover_box(Gamepad &pad);
    
    /** Check box cover cross1 and score
    * @details Check if box cover which means the player get one score.
    *          First, the positon of cross1, box1 and box2 are read. Then, compare both of cross1 to box1
               cross1 to box2. If any of these two comparsion return true, one score will be added
    */
    void box_cover_cross1_score(Gamepad &pad);
    
    /** Check box cover cross2 and score
    * @details Check if box cover which means the player get one score.
    *          First, the positon of cross2, box1 and box2 are read. Then, compare both of cross2 to box1
               cross2 to box2. If any of these two comparsion return true, one score will be added
    */
    void box_cover_cross2_score(Gamepad &pad);

private:

    Box _b1;
    Box _b2;

    // positions of the Boxes
    int _b1x;
    int _b2x;
    int _b1y;
    int _b2y;

    Ppl _ppl;

    // poitions of the ppl
    int _pplx;
    int _pply;

    Cross _c1;
    Cross _c2;

    // position of the crosses
    int _c1x;
    int _c2x;
    int _c1y;
    int _c2y;

    Menu _menu;

    //reading of bottom
    int _bb;
    int _bx;
    int _by;
    int _ba;
    int _bjoy;
    int _bj;

    //indicator for ppl and ppl touching
    int _s; //ppl with b1
    int _r; //ppl with b2
    
    //indicator for holding ppl position
    int _temp; 

    int _score;

};

#endif