#ifndef WDPLAYER_H
#define WDPLAYER_H

/** WDplayer Class
@brief A PWM Audio Player for .Wav Files only for 8-bit Mono-Stereo Audio.
@brief Recomended Sample rate is 11025Hz for Inline Functions and 8000 for Outline functions
@brief Recomended Online Wav converter :https://audio.online-convert.com/convert-to-wav.

@author Saad Tayyab
@date 18th April 2019

@code
//Required Libraries to Work
#include "mbed.h"
#include "SDFileSystem.h"
#include "WDplayer.h"

//Define SD Card
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");

int main() {
    
    //Defining a File Path
    const char *path; 
    path = "/sd/audio.wav";

    //Creating an Object
    WDplayer wav(PTC10);
    
    //Intialize with File Path
    wav.intWD(path,false);

    //Play the File
    wav.WDplay();
   
    //Changing Path
    path = "/sd/audio2.wav";
    wav.intWD(path,false);
    
    //Using the One Tick One Sample Playing
    while(1) {
        wav.WDtck();
        
        //Setting up a Sample Wait Time
        wait(wav.get_pwmfreq() / 2.0f);
    }
    
   //New ISR Outline Functions
   //Change Path and Allow Allocation
   path = "/sd/Game-Files/AudBin/play1.wav";
   wav.intWD(path,true);
   
   //Run Setup and Run ISRpreload Frequently
   wav.ISRset();
   while (1) {
        wav.ISRpreload();     
   }
   
   //Delete Memory and Deattch
   wav.ISRreset();
   
}
@endcode
*/


#include <iostream>
#include "mbed.h"

class WDplayer{
    
    public:
    /** Constructor.
     *@param buzzer The pin name on which the PWM buzzer is connected.
     */    
    WDplayer(PinName buzzer);
    /** Deconstructor.
     */
    ~WDplayer();
    /** Intializes WDplayer with a file path and PWM pin.
     *
     *@param path File path pointer.
     *@param allocate To enable memory allocation for ISR run functions (4KB space).
     */
    void intWD(const char *path, bool allocate);
    
    
    //Inline Functions Pack
    
    /** Plays the Wav file (mbed waits till file has finished playing).
     */
    void WDplay();
    /** Plays the Wav file one sample at a time requires a wait timer.
     */
    void WDtck();
    /** Returns the created PWM period.
     *
     *@returns A float of the period the PWM is working on.
     */
    float get_pwmfreq();
    
    
    //Outline ISR Functions Pack
    
    /** Function to set up ISR ticker and play audio
     */
    void ISRset();
    /** Preloads 8000 samples into memory for ISR Playing (Perfect for 8000 rate sampling) 
     *@brief Must be called once every 0.25 seconds if not it will sound bad. At higher sampling rates call it more frequently.
     *@brief e.g 11025Khz - 0.125s, 16000Khz - 0.05s
     */
    void ISRpreload();
    /** Function realise allocated memory and reset
     */
    void ISRreset();
    /** Function to pause the ISR audio
    */
    void ISRpause();
    /** Function to resume the ISR audio
    */
    void ISRresume();
    
    private:
    //Hardware Pointers
    PwmOut *_dac;
    //Accessable Varibles
    float _pwmfreq;
    //Dynamic Memory Pointers
    unsigned char *cache;
    unsigned char *buffer;
    //Other pointers
    const char *_path;
    FILE *_fptr;
    //Volatile Variables
    volatile int vtck;
    //Private Variables
    unsigned int _length;
    unsigned int _tck;
    bool lock;
    //Internal Functions
    void ISRtck();
    void UpdateBlock1();
    void UpdateBlock2();
};
#endif