#ifndef SCENECREATOR_H
#define SCENECREATOR_H

/** SceneCreator Class
@breif Creates RPG game scenes using the following libraries, CaMove, Bitmap, WDplayer and SceneFunctions.

@author Saad Tayyab
@date 25th April 2019

@code
//Required Libraies
#include "mbed.h"
#include "WDplayer.h"
#include "SDFileSystem.h"
#include "Bitmap.h"
#include "CaMove.h"
#include "SceneCreator.h"

//Creating SD Card Object
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");

//Creating LCD Object
Bitmap lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);

//Creating Charater and Interation Object
CaMove CM(PTB9,PTB11,PTB10);

//Creating Music Player Object
WDplayer wav(PTC10);

//Creating Scene Object
SceneCreator SNC(PTD0);

//Timer for Random Number Generation
Timer Time;

int main() {
    
    //Initializing Music Player
    //Setting Path
    const char *path; 
    path = "/sd/audio.wav";
    wav.intWD(path, true);
    
    //Initializing LCD and starting Timer
    lcd.init();
    Time.start();
    
    //Creating a Game Scene Number 0
    SNC.Scenesp(lcd,CM,0,40,20,Fd,0,wav);
    
    //Generating Random Number based on Time taken for player to complete Scene
    Time.stop();
    int Random = (int)Time.read();
    Random = Random % 3;
    
    //Changing Backgroud Music
    wav.intWD(path, true);
    
    //Creating a Game Scene Number 0
    SNC.Scenesp(lcd,CM,1,35,29,Fd,Random,wav);
}
@endcode
*/

#include "mbed.h"
#include "CaMove.h"
#include "Bitmap.h"
#include "SceneFunctions.h"
#include "WDplayer.h"

class SceneCreator : public SceneFunctions {
        
    public :
    /** Constructor
     *
     *@param Button Pin for button that will be used for interations
     */
    SceneCreator(PinName Button);
    /** Destructor */
    ~SceneCreator();
    /** Function to generate scenes.
     *@brief Function controls and manages the all events and changes in scenes dependent on user actioon in the scene
     *@brief It is divided into two parts the Pre Events and Pos Events 
     *@brief Pos Events (Tempormary Events that occur due to user having interated with an object used for speaking screens or other small animations) 
     *@brief Pre Events (Permanent changes that occur after an Pos Event has been triggered eg a box appearing).
     *@brief Bluilt to allow all types and modes of RPG gameplay in a single algorithm
     *@brief Events can be set in the SceneFunctions Library that is Parent to SceneCreator
     *@brief WDplayer must be intialised before using this Function
     *
     *@param lcd Bitmap object name 
     *@param CM CaMove object name
     *@param spl Scene to spawn number can only be between 0 to 4 (Max 5 Scenes) 
     *@param spx Charater x spawn location on screen at start between 0 to 84
     *@param spy Charater y spawn location on screen at start between 0 to 48
     *@param spd Charater spawn Direction either (Fd, Bd, Rt, Lt)
     *@param rng Random number for Random events
     *@param wav WDplayer object name
     */
    void Scenesp(Bitmap &lcd, CaMove &CM,short int spl,short int spx,short int spy, Direction spd, int rng,WDplayer &wav);
    /** Function to get player status
     *@returns true if dead and false if alive
     */
    bool get_dead();
    /** Function to overide player status
     *@param opt Wanted status of player true is dead and vice versa.
     */
    void set_dead(bool opt);
    /** Function to flash the screen
     *@param lcd Bitmap Object.
     */
    void flashy(Bitmap &lcd);
    
    private :
    //Scene Creator Internal Functions for Scenesp
    void setup(Bitmap &lcd, CaMove &CM,short int spl,short int spx,short int spy, Direction spd, WDplayer &wav);
    void update_charaters(Bitmap &lcd, CaMove &CM,short int spl, WDplayer &wav);
    void dynamic_wait();
    
    //Aiding Functions
    void task_completed(bool &uno, Bitmap &lcd, WDplayer &wav); //Used in Core_2
    
    //Event Cores
    void Core_1(Bitmap &lcd, CaMove &CM,short int spl,int rng,bool p[5]);  //Pre-Engine Core
    void Core_2(Bitmap &lcd, CaMove &CM,short int spl,int rng,bool (&p)[5],WDplayer &wav);  //Pos-Engine Core
    
    //Scene setup function
    void set_sreg(CaMove &CM,int sce); //Used in setup
    
    //pre Event
    void pre_func0(int sce,Bitmap &lcd,CaMove &CM,int rng);
    void pre_func1(int sce,Bitmap &lcd,CaMove &CM,int rng);
    void pre_func2(int sce,Bitmap &lcd,CaMove &CM,int rng);
    void pre_func3(int sce,Bitmap &lcd,CaMove &CM,int rng);
    void pre_func4(int sce,Bitmap &lcd,CaMove &CM,int rng);
    
    //pos Event
    void pos_func0(int sce,Bitmap &lcd,CaMove &CM,int rng,WDplayer &wav);
    void pos_func1(int sce,Bitmap &lcd,CaMove &CM,int rng);
    void pos_func2(int sce,Bitmap &lcd,CaMove &CM,int rng);
    void pos_func3(int sce,Bitmap &lcd,CaMove &CM,int rng);
    void pos_func4(int sce,Bitmap &lcd,CaMove &CM,int rng);
    
    //Private Varibales
    bool dead;
    bool p[5];
    
};

#endif