#ifndef CAMOVE_H
#define CAMOVE_H


#include "mbed.h"
#include "Bitmap.h"
#include <vector>


/** Enum For Charater's Facing Direction. */
enum Direction {
        Lt, /**< Charater Facing Left. */
        Rt, /**< Charater Facing Right. */
        Fd, /**< Charater Facing Forward. */
        Bd  /**< Charater Facing Backward. */
};


/** CaMove Class
@brief Class for Charater Animation and Movement Interation.
@brief Controls Interrupts for Interation Button and The Joystick for Charater Movement
@brief Uses the FilePaths.h to supply File Location Information and uses Bitmap.h to create animations on the LCD

@author Saad Tayyab
@date 2th April 2019

@code
//Required Libraries
#include "mbed.h"
#include "CaMove.h"
#include "Bitmap.h"
#include "SDFileSystem.h"

//Define SD Card
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");

//Define LCD
Bitmap lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);

//Define CaMove
CaMove CM(PTB9,PTB11,PTB10);

//A mini game instance where charater can click two black boxes in either corner
//depending on clicked box the charater will either enter a new instance where he
//can leave the screen and when he does the program ends or the instance where the
//screen fades to black. The program ends with a circle on sceen.

int main() {
    //Initalize and Set Spawn Values
    CM.init(40,20,Fd);
    lcd.init();
    
    //Setting Interative Regions
    CM.set_region(6,6,8,8);
    CM.set_region(60,30,8,8);
    
    //While Loop that ends when Interative Region triggered
    while(!CM.is_trg()) {
        lcd.clear();
        lcd.drawRect(2,2,70,40,FILL_TRANSPARENT);
        lcd.drawRect(3,3,68,38,FILL_TRANSPARENT);
        lcd.drawRect(4,4,8,8,FILL_BLACK);
        lcd.drawRect(62,32,8,8,FILL_BLACK);
       
        //Move the Charater and Update Movement Animation
        CM.move(lcd);
        lcd.refresh();
        wait(0.1);
    }
    lcd.clear();
    //Interative Region 1 Triggered
    if (CM.get_treg() == 0) {
        
        //Delete Interartive Regions
        CM.delete_regions();
        CM.init(40,20,Bd);
        
        //While Loop Dependent on if Charater is In Screen
        while(CM.in_screen()) {
            lcd.clear();
            CM.move(lcd);
            lcd.refresh();
            wait(0.2);
        }
    }
    
    //Interative Region 2 Triggered
    if (CM.get_treg() == 1) {
        for(int i = 0; i < 84; i++) {
            lcd.drawRect(0,0,i,48,FILL_BLACK);
            lcd.refresh();
            wait(0.05);
        }
    }
    //Ending Screen
    lcd.clear();
    lcd.drawCircle(40,20,5,FILL_TRANSPARENT);
    lcd.refresh();   
}
@endcode
*/
class CaMove {
    public :
    /** Constructor
     *
     *@param Button Pin for the button to control interations.
     *@param Pot_h Pin for the joystick horizontal potentiometer.
     *@param Pot_v Pin for the joystick vertical potentiometer.
     *
     */
    CaMove(PinName Button, 
           PinName Pot_h, 
           PinName Pot_v);
    /** Destructor */
    ~CaMove();
    /** Function to initilze the charater spawn point.
     * 
     *@param x X-location on screen to spawn at (0 to 84).
     *@param y Y-location on screen to spawn at (0 to 48).
     *@param D In which direction the charater should spawn.
     *
     */
    void init(int x,int y,Direction D);
    /** Function that initilizes chasing Charater Models. 
     *@param chaser Enter Integer to select chaser (1-Death Zone , 2- Girl (slow), 4-Girl (fast), other-Ghost)
     */
    void AIinit(int chaser);
    

     //Player Functions
     
     /** Function that moves the charater but also manages all animations related to its movement and collisions. Double Black lines cannot be crossed and Single Black lines can only be crossed one way.
     *
     *@param lcd Bitmap Class Object.
     *
     */
    void move(Bitmap &lcd);
    /** Function to check if charater is in the screen.
     *
     *@return Returns true if charater is on the screen otherwise false.
     *
     */
    bool in_screen();
    
    //Interactive Regions Function pack
    
    /** Function to set an interative region (rectangle areas only). Everytime it is run creates an additional new region.
     * 
     *@param xmin the top left x-location of region.
     *@param ymin the top left y-location of region.
     *@param xl Width of region.
     *@param yl Height of region.
     *
     */
    void set_region(int xmin, int ymin, int xl, int yl);
    /** Function to delete all interative regions. */
    void delete_regions();
    /** Function that checks if the charater triggered an interative region (set to run on button press). */
    void intercheck();
    /** Function that checks if an interative region was triggered.
     *
     *@returns Returns true only when an interactive region was triggered for that instance but otherwise remains false. */   
    bool is_trg();
    /** Function that tells which interactive was triggered.
     *
     *@return Returns an integer based on which region was triggered, regions are labelled according to order of declaration in program counts from 0. Will return -1 if no regions triggered.
     *
     */
    int get_treg();
    
    //Chaser Functions
    
    /** Function that spawns the chasing Charater.
     *
     *@param x The x-location where Charater will Spawn.
     *@param y The y-location where Charater will Spawn.
     */
    void spawn(int x, int y);
    /** Function that moves the Charater per Frame.
     *@param lcd The Bitmap class object.
     */
    void chase(Bitmap &lcd, int girl);
    /** Function that tell if the player is caught by chaser Charater.
     *@returns Returns a boolean true is caught and false if not.
     */
    bool is_caught();
    
    private :
    //Hardware Pins
    InterruptIn *_check;
    AnalogIn *_hoz;
    AnalogIn *_ver;
    //Enum declaration
    Direction _fc;
    //Priavte Variables
    int _h;
    int _v;
    short int _itr;
    bool LowerH, UpperH, MidH, LowerV, UpperV, MidV;
    
    std::vector<int> _vreg;  //Vector for storing interactive region x - y locations.
    
    //Volatile variables for ISR Interrupts
    volatile int _treg;
    volatile bool _trg;
    
    //AI Pack Variables
    bool enabled;
    int _ch;
    int _cv;
    int increment;
    
    //Internal Functions
    //Player Internal Functions
    void check_collision(bool left,bool right,bool down,bool up);
    //Interaction Internal Functions
    void set_statements(int Vvalue);
    void check_sides(int Vvalue);
    //Aiding Functions
    void set_treg(int Vvalue);
    //Rendering Functions
    void rend(Bitmap &lcd);
    void render_chaser(Bitmap &lcd,int chaser);
    
};

#endif