#ifndef BITMAP_H
#define BITMAP_H

/** Bitmap Class
@breif A library to load .bmp images to the N5110 LCD
@breif It works on 1-Bit MONO-Chrome Images only
@breif Recommended online converter: https://online-converting.com/image/convert2bmp/
@breif Recommended Size 84x48 pixels 1:1 aspect ratio

@author Saad Tayyab
@date 18th April 2019

@code
//Required Libraries to Work
#include "Bitmap.h"
#include "mbed.h"
#include "SDFileSystem.h"

//Define SD card
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");

//Define LCD Pins
Bitmap lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);

int main() {
    //Create Path to Image File
    const char *path; 
    path = "/sd/img.bmp";
    
    //Intialize the LCD
    lcd.init();
    lcd.clear();
    
    //Render the Image using Bitmap
    lcd.renderBMP(path,0,0);
    lcd.refresh();
}

@endcode
*/

#include "N5110.h"
#include <bitset>
#include <iostream>

class Bitmap : public N5110 {
    
public:
    /** Constructor
     * @param sce  Pin connected to chip enable for N5110 Constructor
     * @param rst  Pin connected to reset for N5110 Constructor
     * @param dc   Pin connected to data/command select for N5110 Constructor
     * @param mosi Pin connected to data input for N5110 Constructor
     * @param sclk Pin connected to serial clock for N5110 Constructor
     * @param led  Pin connected to LED backlight for N5110 Constructor
     */
    Bitmap(PinName const scePin,
           PinName const rstPin,
           PinName const dcPin,
           PinName const mosiPin,
           PinName const sclkPin,
           PinName const ledPin);
    /** Loads the image into LCD Buffer at desired location on LCD.
     *
     * @param path Pointer to image file location.
     * @param x X-location of top left corner of image on LCD.
     * @param y Y-location of top left corner of image on LCD.
     *
     */
    void renderBMP(const char *path ,unsigned int const  x, unsigned int const  y);   
private:
    //Private Variables
    unsigned long offbits;
    unsigned long height;
    unsigned long width;
    int row;  //row counter
    int colomn; // colomn counter
    //Internal Functions
    void readDIB(FILE *bmp);   //reads DIB header for image Information
    void swapEndian(FILE *bmp, unsigned char *buffer, std::bitset<32> &bits); 
    void halfLoop(unsigned const int x, unsigned const int y, std::bitset<32> bits); //Skips packing Bits and prints what's left
    void fullLoop(unsigned const int x, unsigned const int y, std::bitset<32> bits); //prints full DWORD of image
    //Aiding Functions
    void renderDWORD(std::bitset<32> bits , unsigned int const x, unsigned int const y);
    
};

#endif