#ifndef FLOORS_H
#define FLOORS_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Enemy.h"

/** Floor class
@brief Class for the floor
@author Melissa Hartmann
@date May 2019
*/
class Floors
{
public:
    Floors();
    ~Floors();

    /**
    @brief Defines the initial position of the floor, its width and height
    @param int y
    @param int width
    @param int height
    @details The x-coordinate is initially random and excludes the space 30-40 so that the doodler falls
    to a floor that is set to be at the bottom center (instead of a random position) in the engine cpp
    (to make sure there is always a stating point position for the doodler). This is done by using a
    decision object to choose which randomly generated x-coordinate position the doodler will have
    (either at the left side or the right side). The y-coordinate position is inputed from the engine
    cpp and kept the same.
    */
    void init(int y, int width, int height);

    /**
    @brief Prints the floor into the LCD screen
    @param N5110 &lcd
    @details The function draws a 14 x 1 rectangle
    */
    void draw(N5110 &lcd);

    /**
    @brief Updates the position of the floor
    @param float doodler_pos_y is inputed into the function to decides if the floor should move
    or remain at its position.
    @details The function first checks if the position should change, which occurs if the doodler's position is above
    the middle of the screen's height (x-coordinate less than 25). If it is so, the floor will shift downwards (add 1 to
    the y-coordinate). The function then checks if the floor has reached the bottom of the screen (y-coordinate over 45)
    and if so, the floor will re-appear at the top of the screen (y-coordinate = 9) and at a random x-coordinate position,
    making there be 6 floors always on the screen.
    */
    void update(float doodler_pos_x, float doodler_pos_y, float _bullet_pos_x, float _bullet_pos_y);
    
    /**
    @brief Function to check if doodler or bullet have collided with the enemy 
    @param float _bullet_pos_x
    @param float _bullet_pos_y
    @details Gets the current values of doodler (already defined as private variables) and the bullet (inputed to the 
    function) in order to compare it with the position of the enemy. If they collide, the end_game decision variable will
    indicate if the game should end, or if the bullet hit the enemy, the game will continue but the enemy will be erased. 
    */
    void check_enemy(float _bullet_pos_x, float _bullet_pos_y);
    
    /**
    @brief Returns the current floor's position as a vector
    @details Gets the current value in the floor's class for the position
    */
    Vector2D get_position();

    /**
    @brief Sets the floor's position in the Floor's class to equal the inputed vector parameter
    @param Vector2D pos is inputed to define the new position
    @details The function sets the floor's position by making the current position equal the inputed
    vector parameter to the function.
    */
    void set_position(Vector2D pos); // mutators
    
    /**
    @brief Returns a true/false statement that is called on engine to decide if game should end or not
    @details The function can be called in the engine to check if the game should end or not (if the doodler has collided
    with the enemy)
    */
    bool get_end_game();

private:
    Enemy eny;
    int _height;
    int _width;
    Vector2D _position;
    Vector2D enemy_position;
    bool put_enemy;
    bool end_game;
    int place;
    float _doodler_pos_x;
    float _doodler_pos_y;

};
#endif