#ifndef ENGINE_H
#define ENGINE_H

#include <iostream>
#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Floors.h"
#include "Doodler.h"
#include "Bullet.h"

/** Engine class
@brief Class for the engine of the game
@author Melissa Hartmann
@date May 2019
*/
class Engine
{
public:
    Engine();
    ~Engine();

    /**
    @brief Initializing game
    @details The function defines the screen's size, and the initial position of the doodler,
    the bullet and the 6 floors. This is done by inputing the functions of initialization to the
    corresponding class. A for loop is used to repeat the process to create 6 floors, by defining
    them as an array of 6 objects.
    */
    void init();

    /**
    @brief Initializes the screen's size and the initial score
    @details The function defines the screen's position, height and width (which remains the same
    throughout the game). It also sets an initial score of 0.
    */
    void screen_init();

    /**
    @brief Sets the inputed gamepad parameters
    @param Gamepad &pad
    @details The function sets the inputed gamepad parameters for the direction of the joystick,
    its magnitude of inclination and checks if the button Y is pressed.
    */
    void read_input(Gamepad &pad);

    /**
    @brief Prints the doodler, floors and bullet
    @param N5110 &lcd
    @details The function calls the draw functions for the doodler, floors and bullet. For the floors, a
    for loop is used to repeat the process to draw the 6 floors. It also draws the screen rectangle and the score.
    */
    void draw(N5110 &lcd);

    /**
    @brief Updates the classes in the game by calling it's update functions
    @param Gamepad &pad
    @details Firstly, the function checks the collision of the doodler with the floors by getting their current positions
    and inputing them to the "check_floors_collision" function, which will update the velocity of the doodler. Secondly, it
    calls the "update" functions for the floors and doodler which will update their positions. It then calls the
    "add_score" function to update the score value. Next, it checks if the doodler fire's by inputing the doodler's
    previous position and updated position, as well as the pad's input. Finally, it calls a function to check if the doodler
    has reached the bottom of the screen to end the game.
    */
    void update(Gamepad &pad);

    /**
    @brief  Checks a collision and sets the doodler's velocity (if it jumps) direction
    @param Gamepad &pad to make a sound each time the doodler jumps
    @param float doodler_pos_x
    @param float doodler_pos_y
    @param double doodler_vel_y
    @details Checks a collision by comparing the doodler's position with the floors' ones with 3 conditions:
    if the doodler is moving downwards (its velocity is positive), if the doodler's x-coordinate is within the
    floor's x-coordinate and if the floors' y-coordinate is equal or +2/-1 bits of the doodler's feet.
    If the conditions are met, it means there is a collision, which means the doodler's velocity will change direction
    (will be set to a negative value). The function also checks the doodler's velocity function to update the velocity
    accordingly (make it increase or decrease by multiplying it with constant values) so that the doodler accelerates
    or decelerates according to its current velocity.
    */
    void check_floors_collision(Gamepad &pad, float doodler_pos_x, float doodler_pos_y, double doodler_vel_y);

    /**
    @brief Checks if the doodler fires
    @param Gamepad &pad
    @param float updated_doodler_pos_x
    @param float updated_doodler_pos_y
    @param float doodler_pos_y
    @details The function checks the bullet's position and the input of the user (if the
    Y button has been pressed). If the bullet is not currently fired (its position
    equals the current doodler's trunk position) and the button Y is pressed, then the bullet is
    updated to a new position (upwards) and the pad makes a sound indicating it is shooting. If
    it is currently above the doodler's position, it means it has been fired, so continues being
    updated. The update function will eventually check if the bullet has reached the end top of the
    screen, which will make the bullet return to the doodler's trunk's position. If this is the case
    and the button Y has not been pressed, the bullet remains in the doodler's position (by inputing
    the doodler's position to the initialize function instead of the update one).
    */
    void check_fire(Gamepad &pad, float updated_doodler_pos_x, float updated_doodler_pos_y, float doodler_pos_y);
    
    /**
    @brief Increases score by checking the position of the floor
    @details The function uses a foor loop to check the current position of every floor and
    if they have reached the top of the screen (meaning they have re-appeared at the top, the
    score will have one point added. This means that the score depends on how many floors re-appear,
    which represents how high the doodler has moved upwards.
    */
    void add_score();
    
    /**
    @brief Sets the decision statement to end the game depending on the doodler's y-coordinate position
    @param float doodler_pos_y
    @details The function sets the decision statement to end the game by using an if statement dependent on 
    the read doodler's position. If the position has reached the bottom of the screen, the game should end, so
    the decision will be true. 
    */
    void set_game_over(float doodler_pos_y);
    
    /**
    @brief Function called on main.cpp to fix the limitation of the score addition and avoid an infinite addition
    @details The function will subtract one score value within the engine class when called on the main function. It is 
    called if the a floor remains at the same position (that makes it add points to the score) for more than one update.
    */
    void subtract_score();
    
    /**
    @brief Returns the current score
    @details The function returns the current score of the game. It is called in the main.cpp to display 
    the final score on the game over screen.
    */
    int get_score();

    /**
    @brief Returns the decision to fix the score addition of the score limitation. 
    @details The function is called in the main.cpp to check if the score needs to be substracted or not.
    */
    bool  get_score_check();
    
    /**
    @brief Returns the decision to end the game 
    @details The function is called in the main.cpp to check if the user loose the game and depending on the true/false
    statement the main.cpp will display the game over screen.
    */
    bool get_game_over();

private:
////////////////////  OBJECTS  //////////////////////////
    Bullet _b; 
    Floors _f[6]; // array of 6 floors
    Doodler _dood;
    
//////////////////  VARIABLES  //////////////////////////
    int _score;
    bool _score_check;
    bool enemy_collision[6];
    bool game_ends;
    int _screen_pos_x;
    int _screen_pos_y;
    int _screen_width;
    int _screen_height;
    Direction _d;
    float _mag;
    bool _button_Y;  
    int _floors_height;
    int _floors_width;
    Vector2D _f_pos[6];  // x and y positions of the floors
    float  doodler_pos_x;
    float  doodler_pos_y;
    float  updated_doodler_pos_x;
    float  updated_doodler_pos_y;
    float  doodler_vel_x;
    double doodler_vel_y;
    float _b_pos_x;
    float _b_pos_y;
};
#endif