#ifndef DOODLER_H
#define DOODLER_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** Doodler's class
@brief Class for the doodler
@author Melissa Hartmann
@date May 2019
*/
class Doodler
{
public:
    Doodler();
    ~Doodler();
    /**
    @brief Defines the initial position of the doodler
    @param float position_x uses float since the velocity will be added, which is not an integer.
    @param float position_y uses float since the velocity will be added, which is not an integer.
    @param double velocity_y needs to be double in order for it to decelerate to a small value that
    approaches zero.
    @details The intial position of the doodler is at the centre of the screen and the values are
    gotten from the Engine class. It also defines the gravity as a positive vale greater than 1
    and the up object as a negative vale less than 1.
    */
    void init(float position_x, float position_y, double velocity_y);

    /**
    @brief Prints the doodler into the LCD screen
    @param N5110 &lcd
    @details The function draws a sprite of 13 x 15 bits that shows the image of the doodler
    */
    void draw(N5110 &lcd);

    /**
    @brief Updates the position of the doodler
    @param Direction d is gotten from the user input of the joystick and determines the movement of
    the doodler in the x-direction. It moves left/right if the joystick directs left/right at an angle
    or not.
    @param float mag is used to accelerate the movement of the doodler in the x-direction since it
    moves 5 frames * the magnitude of the joystick.
    @details The function calls both the x-coordinate position update function and y-coordinate position update function.
    */
    void update(Direction d, float mag);
    
    /**
    @brief Updates the x-coordinate position of the doodler
    @param Direction d is gotten from the user input of the joystick and determines the movement of
    the doodler in the x-direction. It moves left/right if the joystick directs left/right at an angle
    or not.
    @param float mag is used to accelerate the movement of the doodler in the x-direction since it
    moves 5 frames * the magnitude of the joystick.
    @details It reads the input movement of joystick and defines the velocity_x as a variable dependent on the magnitude 
    of inclination read, so that the velocity changes accoordingly. It uses the direction input to decide in an if statement if the 
    position_x should increase or decrease in value. It then checks the doodler does not leave the screen rectangle (39 x 82) in 
    the x-direction by making it remain in the its position.
    */
    void update_x(Direction d, float mag);
    
    /**
    @brief Updates the y-coordinate position of the doodler    
    @details The function updates the y-coordinate position of the doodler. It has no input paramenters from the gamepad, 
    since it only depends on the floors below it. It checks the direction of the velocity (dependent on the collision
    with the floors) to check the direction of the doodler's movement (depending on if it is jumping or falling). It also 
    checks the doodler remains within the y-coordinates of the screen (only top edge, because the bottom edge means the
    game ends).
    */
    void update_y();

    /**
    @brief Function to check the doodler's y-coodinate velocity
    @details The function checks the current velocity of the doodler and depending on its direction
    (positive or negative) it will decide if the doodler's updated velocity will remain the same or
    change direction (jump). If the velocity is positive, it means it is currently jumping, so it
    should continue doing so but in a decreasing velocity (decelerating), which is why it is
    multiplied times the _up constant (value less than 1). If the velocity is negative it means
    it is falling downwards and increasing in velocity (accelerating). This is done by multiplying the
    current velocity by the gravity constant value (greater than 1). Finally, if the velocity is 0, it
    means it has completely decelerated upwards and so it should begin to fall. To do so, the velocity
    is given the value of gravity (which is a positive value).
    */
    void check_velocity();

    /**
    @brief Returns the current doodler's position in the x-axis
    @details Gets the current value in the doodler's class for the doodler's position in the x-axis
    */
    float get_position_x();

    /**
    @brief Returns the current doodler's position in the y-axis
    @details Gets the current value in the doodler's class for the doodler's position in the y-axis
    */
    float get_position_y();
    
    /**
    @brief Returns the current doodler's velocity in the x-axis
    @details Gets the current value in the doodler's class for the doodler's velocity in the x-axis
    */
    float get_velocity_x();

    /**
    @brief Returns the current doodler's velocity in the y-axis
    @details Gets the current value in the doodler's class for the doodler's velocity in the y-axis. It is a
    double value since the y-velocity is a double type.
    */
    double get_velocity_y();

    /**
    @brief Sets the doodler's position in the Doodler's class to equal the inputed parameters
    @param float pos_x
    @param float pos_y
    @details The function sets the doodler's position in the doodler's class by making the current
    position equal the inputed parameters
    */
    void set_position(float pos_x, float pos_y);

    /**
    @brief Sets the doodler's velocity in the Doodler's class to equal the inputed parameters
    @param float vel_x
    @param float vel_y
    @details The function sets the doodler's velocity in the doodler's class by making the current
    velocity equal to the inputed parameters
    */
    void set_velocity(float vel_x, double vel_y);

private:
    float _position_x;
    float _position_y;
    float _velocity_x;
    double _velocity_y;
    double _gravity; 
    double _up;
};
#endif