#ifndef SKATEBOARDER_H
#define SKATEBOARDER_H

#include "mbed.h"
#include "Gamepad.h"

/** Enum for skater direction */
enum Skate_direction {
  Left, /**< Facing left. */
  Right /**< Facing right. */
};

/** Enum for different sprites */    
enum Sprite_value {
  Skate_right, /**< Moving right sprite. */
  Skate_left, /**< Moving left sprite. */
  Stand_left, /**< Standing and facing left sprite. */
  Stand_right, /**< Standing and facing right sprite. */
  Skate_duck_right, /**< Moving right and ducking sprite. */
  Skate_duck_left /**< Moving left and ducking sprite. */
};

/** Skateboarder Class
* @brief Describes the characteristics of the skateboarder sprite * @author Lewis Wooltorton
* @date March 2019

@code

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Skateboarder.h"
#include <cstdlib>
#include <ctime>

N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad gamepad;
Skateboarder _skater;

int _moving_counter;
int _jump_counter;
int _skater_x; 
int _skater_y;
int _level_condition;
bool _fall_flag;
Skate_direction _skater_direction;
Sprite_value _skater_sprite;
bool _start_platform_flag;

int main() {
  // Initialise external skater values.
  _skater_direction = Left;
  _start_platform_flag = true;  // For printing start text in EngineController.
  _skater.set_reset_flag(false);
  _moving_counter = 0;
  _jump_counter = 20;  // Start game falling onto the platform. 
  _fall_flag = false;
  while(1) {
    
    // Process Y coord.
    if (_fall_flag) {
      _skater.fall(_fall_flag, gamepad);
    } else {
      _skater.set_y_position(false, _jump_counter, _level_condition, 
                             gamepad);  // Not jumping as input flag is false.
    }
    _fall_flag = _skater.get_fall_flag();  // Update fall flag.
    _skater_y = _skater.get_y_position();
    _jump_counter = _skater.get_jump_counter();  // Update jump counter.
    
    // Process X coord.
    _skater.set_x_position_and_sprite(1, 
      _moving_counter, 
      _skater_direction,
      1);  // For input coords (1,1).
    _skater_x = _skater.get_x_position();
    _moving_counter = _skater.get_moving_counter();  // Update moving counter.
        
    //Update the direction.
    _skater_direction = _skater.get_direction();
    
    // Print sprite.
    lcd.drawSprite(_skater_x,_skater_y,17,10,
                (int *)_skater.get_sprite(_skater_sprite)); 
  }  
}
    
@endcode
*/

class Skateboarder {
 public:
  // Constructor and destructor.
  /**
  * @brief Constructor @details Non user specified.
  */
  Skateboarder();
  /**
  * @brief Destructor @details Non user specified.
  */
  ~Skateboarder();
  
  // Mutators.
  /**
  * @brief Sets the X coordinate of the Skateboarder.
  * @param joy_x @details The horizontal value of the joystick
  * @param moving_counter @details The counter that adds the displacement to the X coordinate from the centre position
  * @param @details The current direction the skater is facing from the enum class Skate_direction
  * @param @details The vertical value of the joystick
  */
  void set_x_position_and_sprite(
    float joy_x, 
    int moving_counter, 
    Skate_direction direction, 
    float joy_y);
  /** 
  * @brief Sets the Y coordinate of the Skateboarder.
  * @param jump @details A boolean flag that is true only if the A button is pressed
  * @param jump_counter @details The counter that adds the displacement to the Y coordinate from the current position
  * @param level_condition @details an integer that is 1 if the skateboarder is under / on top of the top platforms 
  * @param &gamepad @details The gamepad object from Gamepad class
  */
  void set_y_position(bool jump, int jump_counter, int level_condition, Gamepad &gamepad);
  /**
  * @brief Sets the reset flag of the Skateboarder.
  * @param flag @details A boolean flag that is assigned to the Skateboarder reset flag
  */
  void set_reset_flag(bool flag);
  
  // Accessors. 
  /**
  * @brief Gets the X coordinate.
  * @returns The X coordinate of the Skateboarder
  */
  int get_x_position();
  /**
  * @brief Gets the moving counter.
  * @returns The moving counter for the Skateboarder
  */
  int get_moving_counter();
  /**
  * @brief Gets the Y coordinate.
  * @returns The Y coordinate of the Skateboarder
  */
  int get_y_position();
  /** 
  * @brief Gets the jump counter.
  * @returns The jump counter for the Skateboarder
  */
  int get_jump_counter();
  /**
  * @brief Gets the fall flag.
  * @returns The fall flag of the Skateboarder
  */
  bool get_fall_flag();
  /**
  * @brief Gets the reset flag.
  * @returns The Reset flag of the Skateboarder
  */
  bool get_reset_flag();
  /**
  * @brief Gets the sprite value.
  * @returns The sprite value of the Skateboarder (from the enum class Sprite_value)
  */
  Sprite_value get_sprite_value();
  /**
  * @brief Gets the skate direction.
  * @returns The Skateboarders direction (from the enum class Skate_Direction)
  */
  Skate_direction get_direction();
  /**
  * @brief Gets the sprite.
  * @returns The Skateboarders sprite (an integer array)
  */ 
  int * get_sprite(Sprite_value sprite);
  
  // Member Methods.
  /**
  * @brief Executes the falling sequence for the Skateboarder.
  * @param fall_flag @details The boolean fall flag of the Skateboarder
  * @param &gamepad @details The gamepad object from Gamepad class
  */
  void fall(bool fall_flag, Gamepad &gamepad);
 
 private:
  void process_x_variables(float joy_x);
  void check_duck(float joy_y);
    
  int _x;
  int _moving_counter;
  int _y;
  int _level_condition;
  int _level;
  int _jump_counter;
  bool _fall_flag;
  bool _reset_flag;
  Skate_direction _skate_direction;
  Sprite_value _sprite_value;    
};
#endif
