#ifndef MENU_H
#define MENU_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "EngineController.h"

/** State for finite state machine struct */
struct State {
  int output; /**< Integer for output number. */
  int next_state[4]; /**< Array of integers for next possible states. */
};

/** Menu Class
* @brief Class to control the overall state of the game 
* @author Lewis Wooltorton
* @date April 2019


@code

#include "N5110.h"
#include "Gamepad.h"
#include "mbed.h"
#include "Menu.h"

N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad gamepad;
Menu menu;

void init_game();

int main() {
  gamepad.init();
  menu.init(); 
  lcd.init(); 
  lcd.normalMode();      
  lcd.setBrightness(0.5); 
  while(1) {
    
    // clear, refresh lcd and run the menu.
    lcd.clear();  
    menu.run(lcd, gamepad);   
    lcd.refresh();
    wait(0.01);  
  }     
}  

@endcode
*/

class Menu {
 public:
  // Constructor and Destructor.
  /**
  * @brief Constructor @details Non user specified.
  */
  Menu();
  /**
  * @brief Destructor @details Non user specified.
  */
  ~Menu();
  
  // Mutators
  /**
  * @brief Initialises the Menu object.
  */ 
  void init();
  /**
  * @brief Runs the menu via a state machine.
  * @param &lcd @details The lcd object from the N5110 class
  * @param &gamepad @details The gamepad object from Gamepad class
  */
  void run(N5110 &lcd, Gamepad &gamepad);
  
 private:
  void set_input(bool start, bool back, bool x);
  void run_game(N5110 &lcd, Gamepad &gamepad);
  void display_menu(N5110 &lcd, Gamepad &gamepad);
  void display_controls(N5110 &lcd, Gamepad &gamepad);
  void output(N5110 &lcd, Gamepad &gamepad);
  void play_tone(Gamepad &gamepad);
  void play_chord_a(Gamepad &gamepad);
  void play_chord_b(Gamepad &gamepad);
  
  int _menu_input;
  EngineController _controller;
  bool _button;
  int _state;
  int _output;
  bool _tone_flag;
  int _tone_counter;
  int _chord_counter;
};
#endif