#ifndef ENGINE_H
#define ENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Skateboarder.h"
#include "Platforms.h"
#include "Coin.h"
#include "Fire.h"
#include <cstdlib>
#include <ctime>

/** Input struct */
struct Input {
  Vector2D coord; /**< Vector 2D for joystick coords */
  bool A_flag; /**< Boolean flag for the A button */
  };

/** Engine Class
* @brief Handles the game mechanics in a game engine: reads inputs, updates game and LCD 
* @author Lewis Wooltorton
* @date March 2019

@code

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Engine.h"

N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad gamepad;
Engine _game_engine;

int _game_counter;
bool _start_platform_flag;
int _speed_divider;
int _player_score;

int main() {
  _game_engine.init();
  while(1) {
    
    // Run all reset checks.
    _game_engine.check_reset(lcd, gamepad);  // Resets the game if skater has 
    // died.
    _start_platform_flag = _game_engine.get_start_platform_flag();
    if (_game_counter == 100) _game_counter = 0;  // Count from 0 to 99.
    
    // Run the engine.
    _game_engine.read_input(gamepad);
    _game_engine.set_level_condition();  // Determines if the skater is under 
    // upper platforms.
    _game_engine.process_y(gamepad);
    _game_engine.process_x(_game_counter);
    _game_engine.process_sprite();
    _game_engine.check_coin_collision(gamepad);
    _game_engine.check_fire_collision(gamepad, _game_counter);
    _game_engine.update_lcd(lcd, _game_counter);
    
    // Update the game counter and player score.
    if (_game_counter % _speed_divider == 0) 
    _game_engine.generate_level(_game_counter);  // Level speed is determined by 
    // speed divider.
    _game_counter++;
    _player_score = _game_engine.get_player_score();
    _speed_divider = int(-0.25*_player_score + 10);  // Speed divider is 
    // dependent on how many coins you have
  }
}

@endcode
*/

class Engine {
 public:
  // Constructor and destructor.
  /**
  * @brief Constructor 
  * @details Non user specified.
  */
  Engine();
  /**
  * @brief Destructor 
  * @details Non user specified.
  */
  ~Engine();
  
  // Mutators.
  /**
  * @brief Initalises the Game Engine object.
  */ 
  void init(); 
  /** 
  * @brief Sets the input variables.
  * @param &gamepad * @details The gamepad object from Gamepad class
  */
  void read_input(Gamepad &gamepad); // call set input
  /**
  * @brief Sets the level condition of the Skateboarder.
  */
  void set_level_condition();
  
  // Accessors.
  /**
  * @brief Gets the start platform flag.
  * @returns The start platform flag for printing the start platform sprites
  */
  bool get_start_platform_flag();
  /**
  * @brief Gets the player score.
  * @returns The amount of coins the player has collected 
  */
  int get_player_score();
  
  // Member Methods.
  /**
  * @brief Processes y coordinate of the Skateboarder.
  * @param &gamepad @details The gamepad object from Gamepad class
  */
  void process_y(Gamepad &gamepad);
  /**
  * @brief Processes x coordinate of the Skateboarder.
  * @param game_counter @details A counter that increments every loop iteration from 0 to 99
  */
  void process_x(int game_counter);
  /**
  * @brief Processes the Skateboarder sprite.
  */
  void process_sprite();
  /**
  * @brief Updates the LCD display.
  * @param &lcd @details The lcd object from the N5110 class
  * @param game_counter @details A counter that increments every loop iteration from 0 to 99
  */
  void update_lcd(N5110 &lcd, int game_counter);
  /**
  * @brief Generates components of the level: fire, platforms, coin and background.
  * @param game_counter @details A counter that increments every loop iteration from 0 to 99
  */
  void generate_level(int game_counter);
  /**
  * @brief Checks if the game needs to be reset and resets if it does.
  * @param &lcd @details The lcd object from the N5110 class
  * @param &gamepad @details The gamepad object from Gamepad class
  */
  void check_reset(N5110 &lcd, Gamepad &gamepad);
  /**
  * @brief Checks if the Skateboarder has collected a coin and increments player score if it has.
  * @param &gamepad @details The gamepad object from Gamepad class
  */
  void check_coin_collision(Gamepad &gamepad);
  /**
  * @brief Checks if the Skateboarder has touched the fire and initiates reset if it has.
  * @param &gamepad @details The gamepad object from Gamepad class
  * @param game_counter @details A counter that increments every loop iteration from 0 to 99
  */
  void check_fire_collision(Gamepad &gamepad, int game_counter);
    
 private: 
  void reset_skater();
  void reset_engine();
  void set_fall_flag();
  void generate_lower_lines();
  void generate_upper_lines();
  void generate_fire(int game_counter);
  void execute_dying_sequence(N5110 &lcd, Gamepad &gamepad);
  void draw_screen_fire(int game_counter, N5110 &lcd);
  
  Input _input;
  Skateboarder _skater;
  Coin _coin;
  Fire _fire;
  int _fire_y;
  int _moving_counter;
  int _jump_counter;
  int _skater_x; 
  int _skater_y;
  int _level_condition;
  bool _fall_flag;
  Skate_direction _skater_direction;
  Sprite_value _skater_sprite;
  Platforms _lower_platforms;
  Platforms _upper_platforms;
  Line _lower_line_1;
  Line _lower_line_2;
  Line _lower_line_3;
  Line _upper_line_1;
  Line _upper_line_2;
  Line _upper_line_3;
  int _length_1;
  int _length_2;
  int _length_3; 
  bool _start_platform_flag;
  bool _coin_collision_flag;
  int _player_score;
  int _speed_divider;
  int _fire_height;
};
#endif

