#ifndef GOLFENGINE_H
#define GOLFENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "SDFileSystem.h"
#include "Gamepad.h"
#include "Ball.h"


/** Golf Engine Class
* @brief Library to run golf game
* @author Ellis Blackford Stroud
* @date May, 2018
*/
class GolfEngine 
{
public:
    /** Constructor */
    GolfEngine();

    /** Destructor */
    ~GolfEngine();
    
    /** Sets values for new game
    * @param the frame rate in Hz
    */    
    void init(int frame_rate);
       
    /** Draws game screen
    * @param the class used to interact with the lcd display
    */ 
    void drawGame(N5110 &lcd);
    
    /** Reads input from gamepad
    * @param the class used to interact with the gamepad
    */ 
    void read_input(Gamepad &pad);
    
    /** Updates ball position according to input
    * @param the class used to interact with the gamepad
    */ 
    void update_ball(Gamepad &pad, N5110 &lcd);
    
    /** Checks if level is over 
    * @param the class used to interact with the lcd display
    * @param the class used to interact with the gamepad
    * @param the class used to interact with the sd card
    */ 
    void check_end_level(N5110 &lcd, Gamepad &pad, SDFileSystem &sd);
    
    /** Prints current level
    * @param the class used to interact with the lcd display
    */ 
    void printLevel(N5110 &lcd);    
    
    /** Resets game over flag */
    void reset_game_over_flag();
    
    /** Returns game over flag
    * @param true if game over and false if now
    */     
    bool get_game_over_flag();
    
private:

    void drawCourseWalls(N5110 &lcd, const WallMap map[], int size);
    void drawHole(N5110 &lcd, Coord hole);
    void file_append(int value, SDFileSystem &sd);
    void new_level(N5110 &lcd, SDFileSystem &sd);
    void c_maj(Gamepad &pad);
    void g_maj7_flash(Gamepad &pad, N5110 &lcd);

    int _level;
    bool _game_over_flag;
    float _mag;
    Vector2D _joy_coord;
    float _angle;
    Ball _ball;    
};

//Array of Levels struct contains all information needed to draw and run all levels 
//Contains coordinates of ball start position, hole position and all walls with types. The game engine uses 
//these coordinates to draw the ball and hole for each level, to check for bounces and to check if the ball is in the hole and the level is over.
//To add new levels enter line and ball/hole coordinates for new map.
const Levels levels[7] =  { 
                   
                    {{15,32}, {68,31},  //level one - //element one is coordinate of ball and element two is coordinate of hole
                    {{LEFT,{9,24},{9,39}},  //wall types with coordinates used to check for bounces
                    {BOTTOM,{9,39},{74,39}}, 
                    {RIGHT,{74,24},{74,39}}, 
                    {TOP,{9,24},{74,24}}},4}, //size used to stop for loops drawing/bounce checking walls that do not exist in the 16 term WallType array
                    
                    {{18,30}, {64,15}, //level two
                    {{LEFT,{9,27},{9,41}}, 
                    {BOTTOM,{9,41},{73,41}}, 
                    {RIGHT,{73,11},{73,41}}, 
                    {BOTTOMRIGHT,{65,41},{73,33}}, 
                    {TOP,{56,11},{73,11}}, 
                    {LEFT,{56,11},{56,27}},
                    {TOP,{9,27},{56,27}}}, 7},
                    
                    {{18,17}, {64,17}, //level  three
                    {{TOP,{9,10},{27,10}}, 
                    {LEFT,{9,10},{9,45}}, 
                    {BOTTOM,{9,45},{73,45}},
                    {BOTTOMRIGHT,{63,45},{73,35}}, 
                    {RIGHT,{73,10},{73,45}}, 
                    {TOP,{56,10},{73,10}}, 
                    {LEFT,{56,10},{56,32}}, 
                    {TOP,{27,32},{56,32}},
                    {RIGHT,{27,10},{27,32}}},  9},
                    
                    {{15,39}, {70,18},  //level four 
                    {{LEFT,{9,31},{9,44}}, 
                    {BOTTOM,{9,44},{53,44}}, 
                    {RIGHT,{53,25},{53,44}}, 
                    {TOP,{9,31},{36,31}},
                    {LEFT,{36,12},{36,31}},
                    {TOPLEFT,{36,20},{44,12}},
                    {TOP,{36,12},{78,12}},
                    {BOTTOM,{53,25},{78,25}},
                    {RIGHT,{78,12},{78,25}}}, 9},
                    
                    {{31,39}, {70,38},  //level five 
                    {{LEFT,{7,18},{7,43}}, 
                    {BOTTOMLEFT,{7,37},{13,43}},
                    {TOPLEFT,{7,24},{13,18}},
                    {BOTTOM,{7,43},{34,43}}, 
                    {RIGHT,{34,26},{34,43}},
                    {TOP,{7,18},{76,18}},
                    {BOTTOM,{34,26},{51,26}},
                    {LEFT,{51,26},{51,43}}, 
                    {BOTTOM,{51,43},{76,43}},
                    {RIGHT,{60,26},{60,35}},
                    {BOTTOM,{60,26},{76,26}},
                    {TOP,{60,35},{76,35}},
                    {RIGHT,{76,35},{76,43}},
                    {RIGHT,{76,18},{76,26}}}, 14},
                    
                    {{31,40}, {72,29},  //level six 
                    {{LEFT,{7,18},{7,43}}, 
                    {BOTTOM,{7,43},{34,43}},
                    {TOP,{7,18},{34,18}},
                    {RIGHT,{34,18},{34,27}},
                    {RIGHT,{34,34},{34,43}},
                    {TOP,{34,27},{50,27}},
                    {BOTTOM,{34,34},{50,34}},
                    {LEFT,{50,18},{50,27}},
                    {LEFT,{50,34},{50,43}},
                    {TOP,{50,18},{77,18}},
                    {BOTTOM,{50,43},{77,43}},
                    {RIGHT,{77,18},{77,43}},
                    {RIGHT,{60,25},{60,36}},
                    {BOTTOM,{60,25},{67,25}},
                    {LEFT,{67,25},{67,36}},
                    {TOP,{60,36},{67,36}}}, 16},
                    
                    {{41,20}, {41,33},  //level seven 
                    {{LEFT,{10,9},{10,45}}, 
                    {TOPLEFT,{10,14},{15,9}},
                    {TOP,{10,9},{73,9}},
                    {TOPRIGHT,{68,9},{73,14}},
                    {RIGHT,{73,9},{73,45}},
                    {BOTTOM,{10,45},{73,45}},
                    {BOTTOM,{34,23},{50,23}},
                    {RIGHT,{50,15},{50,23}},
                    {BOTTOM,{50,15},{64,15}},
                    {LEFT,{64,15},{64,39}},
                    {TOP,{50,39},{64,39}},
                    {RIGHT,{50,31},{50,39}},
                    {TOP,{34,31},{50,31}},
                    {LEFT,{34,31},{34,39}},
                    {TOP,{19,39},{34,39}},
                    {RIGHT,{19,15},{19,39}},                    
                    {BOTTOM,{19,15},{34,15}},
                    {LEFT,{34,15},{34,23}}}, 18}
};
#endif