#ifndef BALL_H
#define BALL_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** Enum for wall types */
enum WallType { 
  
    LEFT,  /**< left wall */
    RIGHT,       /**< right wall */
    TOP,      /**< top wall */
    BOTTOM,       /**< bottom wall */
    BOTTOMLEFT,  /**< bottom left 45 deg wall */
    BOTTOMRIGHT,  /**< bottom right 45 deg wall */
    TOPLEFT,  /**< top left 45 deg wall */
    TOPRIGHT,  /**< top right 45 deg wall */
    
};

/** Pixel Coordinate Struct*/
struct Coord {
    
    int x; /**< coordinate of x pixel */
    int y; /**< coordinate of y pixel */
    
};

/** Wall Information struct */
struct WallMap {
    
    WallType wall;  /**< wall type */
    Coord start; /**< coordinate of line start */
    Coord end; /**< coordinate of line end */
    
};

/** Levels Information struct */
struct Levels {
    
    Coord ball; /**< start position of ball */
    Coord hole; /**< position of hole */
    WallMap walls[18]; /**< array of walls */
    int wall_count; /**< number of walls in level */
    

};

/** Ball Class
* @brief Class for controlling a  ball
* @author Ellis Blackford Stroud
* @date May, 2018
*/
class Ball 
{
public:
    /** Constructor */
    Ball();
    
    /** Destructor */
    ~Ball();
    
    /** Resets ball private variables for new level
    * @param a structing storing the start coordinate of the ball 
    */    
    void init(Coord start_pos);
    
    /** Draws ball in position
    * @param the class used to interact with the lcd display
    */ 
    void drawBall(N5110 &lcd);
    
    /** Prints shot count 
    * @param the class used to interact with the lcd display
    */ 
    void printShotCount(N5110 &lcd);
    
    /** Draws power bar
    * @param the class used to interact with the lcd display
    * @param a float in the range 0.0 - 1.0
    */ 
    void drawPower(N5110 &lcd, float mag);
    
    /** Draws aim on ball
    * @param the class used to interact with the lcd periphal 
    * @param x and y floats of joystick coordinate in the range -1.0 - 1.0 
    */ 
    void drawAim(N5110 &lcd, Vector2D joy_coord ,float angle);
    
    /** Moves ball position */   
    void move_ball();
    
    /** Checks if ball is shot and if so changes ball velocity to direction and magnitude of joystick
    *   @param the class used to interact with the gamepad 
    *   @param a struct storing mapped coordinates of the joystick position
    */
    void check_shoot_ball(Gamepad &pad, Vector2D joy_coord);
           
    /** Sets total shout count
    *   @param an integer storing the total shot count
    */ 
    void set_total_shot_count(int total_shot_count);
    
    /** Gets total shout count
    *   @return the total shot count
    */ 
    int get_total_shot_count();
    
    /** Gets current level shout count
    *   @return the shot countof current level
    */ 
    int get_shot_count();
        
        
    /** Checks if ball is in hole
    *   @return true if ball is in the hole and false if not
    */ 
    bool check_hole(Coord hole);
    
    /** Checks if ball has met any bounce conditions
    *   @param array of WallMap struct values storing wall types and coordinates
    *   @param integer used to control loop which runs bounce check
    */ 
    void check_wall_bounce(const WallMap map[], int size);
    
    /** Sets frame rate
    *   @param frequency of frame rate in Hz
    */ 
    void set_frame_rate(int frame_rate);
    
    /** Gets bounce flag state
    *   @return true for bounce false for none
    */ 
    bool get_bounce_flag();
    
    /** Reset bounce flag state */ 
    void reset_bounce_flag();

private:
    
    void left_bounce(Coord start, Coord end);
    void right_bounce(Coord start, Coord end);
    void top_bounce(Coord start, Coord end);
    void bottom_bounce(Coord start, Coord end);
    void bottom_left_bounce(Coord start, Coord end);
    void bottom_right_bounce(Coord start, Coord end);
    void top_left_bounce(Coord start, Coord end);
    void top_right_bounce(Coord start, Coord end);
    
    float _x_pos; 
    float _y_pos;
    float _x_vel;
    float _y_vel;
    bool _bounce_flag;
    int _shot_count;
    int _total_shot_count;
    int _frame_rate;
    Direction _direction;
    
};  

#endif