#ifndef STARS_H
#define STARS_H

/**
 * Stars Class
 * @brief Manages, updates and draws the background stars.
 * @author Dmitrijs Griskovs
 * @date 15/04/2019
 */
class Stars {
public:
    /**
     * @var static const int max_small_stars variable;
     * @brief Sets the limit of maximum small stars allowed on the screen.
     */
    static const int max_small_stars = 10;
    /**
     * @var static const int max_medium_stars variable;
     * @brief Sets the limit of maximum medium stars allowed on the screen.
     */
    static const int max_medium_stars = 5;
    
    GameObject small_stars[max_small_stars];
    GameObject medium_stars[max_medium_stars];
    
    /** 
     * @brief draws small stars on the screen.
     * @details when a small star is active, this function updates the position and
     * draws it on the screen. Also, When the star leaves the screen limits it
     * deactivates the star for a new star to fly.
     */
    void updateAndDrawSmallStars(){
        for (int i = 0; i < max_small_stars; ++i) {
            if (small_stars[i].active) {
                small_stars[i].pos.x -= small_star_speed;
                if (small_stars[i].pos.x <= 0){
                    small_stars[i].active = false;   
                }       
                drawSpriteOnTop(small_stars[i].pos, small_star_sprite);
            }
        }        
    }
   
    /**
     * @brief draws medium stars on the screen.
     * @details when a medium star is active, this function updates the position and
     * draws it on the screen. Also, When the star leaves the screen limits it
     * deactivates the star for a new star to fly.
     */
    void updateAndDrawMediumStars(){
        for (int i = 0; i < max_medium_stars; ++i) {
            if (medium_stars[i].active) {
                medium_stars[i].pos.x -= medium_star_speed;
                if (medium_stars[i].pos.x <= -2){
                    medium_stars[i].active = false;   
                }
                drawSpriteOnTop(medium_stars[i].pos, medium_star_sprite);
            }
        }
    }
    
    /** @brief A separate function for delaying the stars spawn time.*/
    void starsSpawnDelay(){
        if  (stars_delay == stars_delay_max){ 
            //This is dealy between stars generation.
            newSmallStarFlies();
            newMediumStarFlies();
            stars_delay = 0;
        }
        else {
            stars_delay += 1;
        }   
    }
    /**
     * @var int stars delay;
     * @brief The value is incremented every time until it reaches stars_delay_max
     * value, which triggers the stars spawn and then is reseted to zero.  .
     */
    int stars_delay;
    
    
      
private:
    /**
     * @var static const int small_star_speed variable;
     * @brief Sets the speed of the small stars.
     */
    static const int small_star_speed = 2;
    /**
     * @var static const int medium_star_speed variable;
     * @brief Sets the speed of the medium stars.
     */
    static const int medium_star_speed = 4;
    /**
     * @var static const int stars_delay_max variable;
     * @brief Sets the limit of the stars delay spawn.
     */
    static const int stars_delay_max = 5;
    
    /** 
     * @brief Makes a small star active and gives it the start positions.
     * @details Searches through the small stars array and if a star is not active,
     * it becomes active and is given the position of x (which is 0) and.
     * and random position of y.
     */
    void newSmallStarFlies() {
        // Search the array of stars if inactive we can use it. - the same as with blasts
        int found = -1;
        for (int i = 0; i < max_small_stars; ++i) {
            if (!small_stars[i].active) {
                found = i;
                break;
            }
        }
        
        if (found != -1) {
            small_stars[found].active = true;
            small_stars[found].pos.x = screen_width;
            small_stars[found].pos.y = rand() % screen_height + game_area_y;
        }
    }
    
     /** 
     * @brief Makes a medium star active and gives it the start positions. 
     * @details Searches through the medium stars array and if a star is not active,
     * it becomes active and is given the position of x (which is 0) and.
     * and random position of y.
      */
    void newMediumStarFlies() {
        // Search the array of stars if inactive we can use it. - the same as with blasts
        int found = -1;
        for (int i = 0; i < max_medium_stars; ++i) {
            if (!medium_stars[i].active) {
                found = i;
                break;
            }
        }
        if (found != -1) {
            medium_stars[found].active = true;
            medium_stars[found].pos.x = screen_width;
            medium_stars[found].pos.y = rand() % screen_height + game_area_y;
        }
    }  
};
#endif