#ifndef PLAYER_H
#define PLAYER_H

/**
 * Player Class
 * @brief Manages player's ship.
 * @author Dmitrijs Griskovs
 * @date 15/04/2019
 */
class Player : public GameObject {
public:
    /**
     * @var static const int max_player_blasts variable;
     * @brief Sets the limit of maximum player blasts on the screen.
     */
    static const int max_player_blasts = 5;
    
    
    GameObject blasts[max_player_blasts];
    CircleBounds player_bounds;
    CircleBounds blast_bounds;
    Game force_shield_check;
    /** 
     * @brief A constructor of the player's ship.
     * @details A constructor for the palyer's sprite body circle area and the
     * blast circle area. It sets the circle radius for collsion callculations.
     */
    Player() {
        player_bounds.center.x = 5;
        player_bounds.center.y = 7;
        player_bounds.radius = 5;
    
        blast_bounds.center.x = 0;
        blast_bounds.center.y = 1;
        blast_bounds.radius = 1;
    }

    /** 
     * @brief Draws the player's blasts on the screen
     * @details This function Will draw every activated blast to the left with the blast_speed.
     * It will deactivate blasts when they leave the screen, for future reuse.
     * If the blast does miss the enemy and leaves the screen limits, the function will
     * substract 10 points in game_score.
     */
    void updateAndDrawBlasts() {
        for (int i = 0; i < max_player_blasts; ++i) {
            if (blasts[i].active) {
                blasts[i].pos.x += blast_speed;
                if (blasts[i].pos.x >= screen_width){
                    blasts[i].active = false;
                    GameGlobals::game_score -= 10;
                }
                drawSprite(blasts[i].pos, blast_sprite);
            }
        }
    }
    
    /** 
     * @brief Makes a blast active and gives it the start positions.
     * @details This function searches the array for the inactive blasts,
     * If a blast is not active, it will set it to active and will start drawing
     * it accross the screen until it reaches the LCD border line.
     */
    void fireNewBlast() {
        // Search the array of blasts if inactive we can use it.
        int found = -1;
        for (int i = 0; i < max_player_blasts; ++i) {
            if (!blasts[i].active) {
                found = i;
                break;
            }
        }   
        if (found != -1) {
            blasts[found].active = true;
            blasts[found].pos.x = pos.x + spaceship1_width - 2;
            blasts[found].pos.y = pos.y + (spaceship1_height/2);
        }
    }
        
    /** 
     * @brief Updates and draws player's ship (including with force shield sprite) positon.
     * @details The function reads the analog input signal from joystick and
     * moves the player's ship on the LCD accordingly.(move joystick, the ship 
     * moves up, move joystick right, the ship moves right). Also, this function checks
     * whether the force shield was activated in game.cpp, if it was then the player's
     * ship is replaced the exactky the same sip but with the added force shield. 
     */
    void updateAndDraw() {
        if(x_dir.read() > joy_threshold_max_x) {
           pos.x -= ship_speed;
        } else if(x_dir.read() < joy_threshold_min_x) {
           pos.x += ship_speed;
        }
        if(y_dir.read() > joy_threshold_max_y) {
           pos.y -= ship_speed; 
        } else if(y_dir.read() < joy_threshold_min_y) {
           pos.y += ship_speed; 
        }
        shipMovementLimits();
        draw();
    }
    /** 
     * @brief Updates and draws player's ship and the force shield sprites. 
     */
    void draw() {
        if (force_shield_check.forceShildActivate()){
            drawSpriteOnTop(pos, player_spaceship1_shield_sprite);   
        } else { 
            drawSpriteOnTop(pos, player_spaceship1_sprite);
        }
    }
private:
    /**
     * @var static const int ship_speed variable;
     * @brief Sets the speed of the player's ship.
     */
    static const int ship_speed = 2;
    /**
     * @var static const int blast_speed variable;
     * @brief Sets the speed of the player's blast.
     */
    static const int blast_speed = 5;
    
    /** Prevents the player's ship to go beyond the playing zone limits.*/
    void shipMovementLimits(){
        // Limits player ship on screen
        if (pos.x < game_area_x){ pos.x = game_area_x;}     
        if (pos.y < game_area_y) { pos.y = game_area_y;}
        int max_player_x = game_area_x + game_area_width - spaceship1_width;
        int max_player_y = game_area_y + game_area_height - spaceship1_height;
        if (pos.x > max_player_x) pos.x = max_player_x;
        if (pos.y > max_player_y) pos.y = max_player_y;
    }  
};
#endif