#ifndef ENEMIES_H
#define ENEMIES_H

/** 
  * Enemy Class
  * @brief A class to describe the states of an enemy ship.
  * @author Dmitrijs Griskovs
  * @date 15/04/2019
  */
class Enemy : public GameObject {
public:
    Enemy() {
    /** A bool that changes states between whehter to go up or down in y-direction.*/
    switch_enemy_y_dir = false;
    enemy_speed = 1;
    }
    /** 
     * @brief Enemy spawn function.
     * @details This function spawns an enemy on the right side of the screen 
     * and at random position in the y-direction.
     */
    void spawn() {
        Point spawn_pos(screen_width, game_area_y + rand() % (game_area_height - enemy2_height));
        GameObject::spawn(spawn_pos);
        dead = false;
        dead_counter = 0;
        blast_countdown = 0;
    }
    /** 
     * @brief enemy death sound and status reset.
     * @details Marks enemy as dead, so it can display explosion anation.
     */
    void die() {
        gamepad.tone(123,0.3);
        dead = true;
        dead_counter = 6;
    }

    /** 
     * @brief Draws and updates enemy sprites on the screen and substracts the score
     * if the ship leaves the screen limits.
     *
     * @details The function draws this enemy ship on the screen, and 
     * changes their x and y positions each time the function is called.
     * If ship is dead then it draws enemy explosion animation and then disables it.
     * Disables the ship when it leaves the screen limits and substracts 
     * the in game score by -50.
     */    
    void updateAndDraw() {
        pos.x -= enemy_speed;
        if (GameGlobals::game_score >= 200){
            if (!switch_enemy_y_dir){pos.y -= 1;}
            else if(switch_enemy_y_dir){pos.y += 1;}
            if (pos.y >= (game_area_height - enemy2_height)){
                switch_enemy_y_dir = false;
            }
            else if (pos.y <= game_area_y){
                switch_enemy_y_dir = true;   
            }
        }
        if (!dead) { drawSprite(pos, enemy2_sprite);} 
        else { updateAndDrawDeathExplosion();}
        
        if (pos.x < 0) {
            GameGlobals::game_score -= 50;
            active = false;
        }
    }
    
    /**
     * @var bool dead 
     * @brief sets if to true when enemy is hit to draw explossion.
     */
    bool dead;
    /**
     * @var int dead_counter 
     * @brief For drawing small explosion animatioin.
     */
    int dead_counter;
    /**
     * @var int blast_countdown 
     * @brief A variable for delaying the enemy shots.
     */
    int blast_countdown;
    /**
     * @var int enemy_speed 
     * @brief How fast should an enemy move.
     */
    int enemy_speed;
private:/** 
     * @brief Draws enemy explosion sprite
     * @details Draws the enemy ships explosion sprites when the player's 
     * shot hits them or when the player collides with them.
     * the animation consists of two sprites "Exploded" and "Half exploded".
     */        
    void updateAndDrawDeathExplosion() {
        if (dead_counter > 0) {
            dead_counter--;
            if (dead_counter > 2) {    
                drawSprite(pos, enemy2_half_exploded_sprite);
            } else {
                drawSprite(pos, enemy2_exploded_sprite);
            }
        } else {
            active = false;
        }
    }   

    bool switch_enemy_y_dir;
};

    
/** 
 * Enemies Class
 * @brief The class manages all enemy ships.
 * @details Responsible for spawning new ships and holding and updating them 
 * and controlling time intervals between enemy shots.
 * @author Dmitrijs Griskovs
 * @date 15/04/2019
 */  
class Enemies {
public:
    /**
     * @var static const int max_enemies
     * @brief Maximum enemies allowed on the screen 
     */
    static const int max_enemies = 4;
    /** 
     * @var static const int max_enemy_blasts
     * @brief Maximum enemies' blasts allowed on the screen
     */
    static const int max_enemy_blasts = max_enemies*2;
    
    Enemy enemy_blasts[max_enemy_blasts];
    Enemy enemies[max_enemies];
    CircleBounds enemy_bounds;
    CircleBounds enemy_blast_bounds;
    
    /** 
     * Constructor 
     * Sets values for the enemy's sprite body circle area, the blast circle 
     * area and the circle radius for collsion callculations.
     */
    Enemies () {
        enemy_bounds.center.x = 5;
        enemy_bounds.center.y = 3;
        enemy_bounds.radius = 6;

        enemy_blast_bounds.center.x = 0;
        enemy_blast_bounds.center.y = 1;
        enemy_blast_bounds.radius = 1;
        
        enemy_blast_speed = 2;
    }
 
    /** 
     * @brief This function spawns a new enemy on the screen.
     * @details It spawns a new enemy on the screen when there is a free space
     * in the enemy aray.
     */  
    void spawnNewEnemy() {
        int found = -1;
        for (int i = 0; i < max_enemies; ++i) {
            if (!enemies[i].active) {
                found = i;
                break;
            }
        }
        
        if (found != -1) {
            enemies[found].spawn();
        }
    }

    /** 
     * @brief Draws the ships and the blasts.
     * @details Updates and draws all active enemy ships and decides when they
     * should fire blasts.
     */  
    void updateAndDrawEnemies() {
        for (int i = 0; i < max_enemies; ++i) {
            if (enemies[i].active){
                enemies[i].updateAndDraw();
                if (!enemies[i].dead) {
                    // Spawn blast on enemy if counter is ready
                    enemies[i].blast_countdown -= 1;
                    if (enemies[i].blast_countdown <= 0) {
                        bool fired = fireEnemyBlast(enemies[i]);
                        if (fired) {
                            enemies[i].blast_countdown = 20;
                        }
                    }
                }
            }
        }
    }

   
    /** 
     * @brief Uodates and draws all active blasts.
     */
    void updateAndDrawEnemyBlasts() {
        for (int i = 0; i < max_enemy_blasts; ++i) {
            if (enemy_blasts[i].active) {
                enemy_blasts[i].pos.x -= enemy_blast_speed;
                if (enemy_blasts[i].pos.x <= -1){
                    enemy_blasts[i].active = false;
                }
                drawSprite(enemy_blasts[i].pos, blast_sprite);
            }
        }
    }
    /**
     * @var int blast_speed 
     * @brief ow fast should an enemy blast moves.
     */
    int enemy_blast_speed;
    
private:
     /** 
     * @brief Spawns a blast at the position of given enemy.
     * @param enemy (const Enemy&)
     */
    bool fireEnemyBlast(const Enemy& enemy) {
        int found = -1;
        for (int i = 0; i < max_enemy_blasts; ++i) {
            if (!enemy_blasts[i].active) {
                found = i;
                break;
            }
        }
        if (found != -1) {
            enemy_blasts[found].active = true;
            enemy_blasts[found].pos.x = enemy.pos.x;
            enemy_blasts[found].pos.y = enemy.pos.y + enemy2_height / 2;
            gamepad.tone(500,0.1);
            return true;
        }
        return false;
    }
};

#endif