#ifndef COLLISION_LIB_H
#define COLLISION_LIB_H

#include <math.h>

/**
 * @struct Point
 * @brief Position or vector on the screen. Supports basic vector arithmetics.
 */
struct Point {
    /*@{*/
    int x; /**< the x position */
    int y; /**< the y position */
    /** Default constructor */
    Point() : x(0), y(0) {}
    /** Constructor with x and y parameters */
    Point(int _x, int _y) : x(_x), y(_y) {}
    
    /** 
     * @brief Vector addition. 
     */
    Point operator+ (const Point& rhs) const {
        Point result(x + rhs.x, y + rhs.y);
        return result;
    }

    /** 
     * @brief Vector substraction. 
     */
    Point operator- (const Point& rhs) const {
        Point result(x - rhs.x, y - rhs.y);
        return result;
    }
    
    /** 
     * @brief Length of vector squared. 
     */
    int lengthSquared() {
        return x * x + y * y;
    }
};

/** 
 * CicleBounds Class
 * @brief Class to define object's bounds as a circle with center(e.g. offset).
 * @author Dmitrijs Griskovs
 * @date 15/04/2019
 */
class CircleBounds {
public:
    /** Default constructor */
    CircleBounds() {};
    /** Constructor with center and radius parameters */
    CircleBounds(Point _center, float _radius): center(_center), radius(_radius) {};
    Point center;
    float radius;
};

/** 
 * @brief Determines whether the two circles collide.
 * @details the function takes in position and circle bounds (center and radius)
 * of two objects and returns wether they collided.
 *
 * @param pos_a positon of x and y of the first object. (const Point&).
 * @param bounds_a center point and radius of the first object. (const CircleBounds&).
 * @param pos_b positon of x and y of the second object. (const Point&).
 * @param bounds_b center point and radius of the second object. (const CircleBounds&).
 * @return true if the two circles overlap.
 */
inline bool circleCollideTwoObjects(
    const Point& pos_a, const CircleBounds& bounds_a, 
    const Point& pos_b, const CircleBounds& bounds_b
) {
    Point center_a = pos_a + bounds_a.center;
    Point center_b = pos_b + bounds_b.center;
    Point difference = center_a - center_b;
    return difference.lengthSquared() <= pow(bounds_a.radius + bounds_b.radius, 2);
}

#endif