#include "mbed.h"
#include "N5110.h"
#ifndef FACE_H
#define FACE_H
#include "Face.h"
#endif
#ifndef CUBE_H
#define CUBE_H
#include "Cube.h"
#endif

/** Face class
*@brief A class used to instantiate a Renderer object, this class provides an interface from the game to the display. Each Face of all the Cube objects will be passed to the Renderer
 for it to be rasterised and its outlines drawn.
*@author Christopher Doel
*@date April, 2019
*/
class Renderer {
  private:
    float fov;
    Cube selectionCube;
    /** Projects the 3D x coordinate to 2D perspective
    *@param The 3d x coordinate as a float
    *@param The 3d z coordinate as a float
    *@returns The 2D perspective fo the x coordinate
    */
    float xTo2D(float x, float z);
    /** Projects the 3D y coordinate to 2D perspective
    *@param The 3d y coordinate as a float
    *@param The 3d z coordinate as a float
    *@returns The 2D perspective of the y coordinate
    */
    float yTo2D(float y, float z);
  public:
    Renderer();
    /** Initialises the LCD and sets the field of view
    */
    void init();
    /** Draws the games horizon line
    *@param The angle of the horizon determined by the joystick position
    */
    void drawHorizon(float angle);
    /** Draw an individual face
    *@param The face object
    *@param The angle of the face object as a float
    */
    void drawFace(Face *face, float angle);
    /** Draw an individual face
    *@param The face object
    *@param The angle of the face object as a float
    */
    void rasterizeFace(float (&points)[4][3], Face *face);
    /** Draw an individual face
    *@param The pointer to an array containing the verticies of the face
    *@param The integer difference between the top horizontal edge x coordinates
    *@param The integer difference between right vertical edge y coordinates
    *@param The integer difference between bottom horizontal edge x coordinates
    *@param The integer difference between left horizontal edge y coordinates
    */
    void drawFillLines(float (&points)[4][3], int diffX2, int stepBottomY,
    int stepTopX, int stepTopY);
    /** Draw the outline of the face
    *@param The pointer to an array containing the verticies of the face
    */
    void drawFaceOutline(float (&points)[4][3]);
    /** Draw all faces within a face array
    *@param The pointer to an array of face objects
    *@param The number of faces as an integer
    *@param The angle of the faces as a float
    */
    void drawAllFaces(Face *faceArray, int noOfFaces, float angle);
    /** Checks whether any area of the face is on the screen
    *@param The pointer to an array containing the verticies of the face
    */
    bool checkOnScreen(float (&Points)[4][3]);
    /** Prints a string at a desired location on the screen
    *@param The pointer to an array of characters of the string
    *@param The x coordinate to draw the string as an integer
    *@param The y coordinate to draw the string as an integer
    */
    void print(const char *text, int x, int y);
    /** Prints a score at a desired location on the screen
    *@param The value of the score as an integer
    *@param The x coordinate to draw the score as an integer
    *@param The y coordinate to draw the score as an integer
    */
    void printScore(int score, int x, int y);
    /** Clears the display
    */
    void clear();
    /** Refreshes the display
    */
    void refresh();
    /** Turns off the display
    */
    void turnOff();
    /** Draws the screen after the user has collided with a cube
    *@param The selected option as an integer
    *@param The high score as an integer
    */
    void setContrast(float contrast);
    void drawDeathScreen(int selection, int highScore);
    /** Draws the death menus buttons
    */
    void drawDeathButtons();
    /** Draws the games home screen
    *@param The selected option as an integer
    */
    void drawHomeScreen(int selection);
    /** Draws the home menus buttons
    */
    void drawHomeButtons();
    /** Draws the cube which indicates the selected button on the menus
    *@param The x coordinate of where to draw the cube as an integer
    *@param The y coordinate of where to draw the cube as an integer
    *@param The z coordinate of where to draw the cube as an integer
    *@param The rotation speed multiplier of the cube as an integer
    */
    void drawSelectionCube(int x, int y, int z, int rotationSpeed);
    /** Draws the first help screen
    */
    void drawHelpScreen1();
    /** Draws the second help screen
    */
    void drawHelpScreen2();
    /** Draws the third help screen
    */
    void drawHelpScreen3();
    /** Draws the third help screen
    */
    void drawHelpScreen4();
};

