#ifndef FACE_H
#define FACE_H
#include "Face.h"
#endif
#include "Renderer.h"
#ifndef CUBE_H
#define CUBE_H
#include "Cube.h"
#endif
#include "Gamepad.h"
#include "mbed.h"
#include "SDFileSystem.h"
 
/**@brief Input struct.
* A struct for holding the infomation regarding the joystick x axis position, a, b, and y buttons and whether the buttons have been pressed within the last 0.2 seconds*/
struct Input{
    float x; /**<The x axis of the joystick*/
    bool yButton; /**<Whether the y button is pressed*/
    bool aButton; /**<Whether the a button is pressed*/
    bool bButton; /**<Whether the b button is pressed*/
    bool yCooldown; /**<Whether 0.2 seconds has elapsed since the y button was pressed*/
    bool aCooldown; /**<Whether 0.2 seconds has elapsed since the a button was pressed*/
    bool bCooldown; /**<Whether 0.2 seconds has elapsed since the b button was pressed*/
};

/**@brief MenuSelections struct.
* A struct for storing the current menu selecitons in both the home and death screen menus */
struct MenuSelections{ //stores the current selections in both menus
    int deathMenuSelection; /**< Which item in the death screen has been selected*/
    int homeMenuSelection; /**< Which item in the home screen has been selected*/
};


/** Game class
*@brief A class used to instantiate a game object, this is used to run the game and consists of the main game loop.
*@author Christopher Doel
*@date April, 2019
*/


class Game {
  private:
    Input input;
    MenuSelections menuSelections;
    
    int noOfCubes;
    bool playing;
    int score;
    int highScore;
    bool inHomeMenu;
    int helpScreenNumber;
    
    FILE *filePointer;
    
    Cube cubeArray[25]; //Store cubes
    Face faceArray[150]; //Store all cubes faces

    Ticker disableA;
    Ticker disableB;
    Ticker disableY;
    Gamepad gamepad;
    Renderer renderer;
    /** Reads the highscore stored in the SD card
    *@returns the highscore as an integer
    */
    int readHighScore();
    /** Writes the new highscore to the SD card
    *@param the highscore as an integer
    */
    void writeHighScore(int score);
    /** generates initial positions for all cubes
    */
    void resetScene();
    /** Increments the score by 1
    */
    void addScore();
    /** Resets the score to 0
    */
    void resetScore();
    /** Checks whether a cube needs to be despawned, if it does then the cube will be translated to the far side of the map
    *@param A pointer to a cube object
    */
    void checkDespawn(Cube *cube);
    /** Checks whether a cube is too close to the user and therefore a collision has occured, the game will be stopped if true
    *@param A pointer to a cube object
    */
    void checkDeath(Cube *cube);
    /** Adds the cubes faces to the array of faces to be passed to the renderer
    *@param A pointer to a cube object
    *@param The integer index of the cubes position in the cube array
    */
    void cubeToBeRendered(Cube *cube, int cubeIndex);
    /** Translates the cube in the z axis towards the user (speed dependant on score) and left and right depending on the joystick location
    *@param A pointer to a cube object
    */
    void moveCubes(Cube *cube);
    /** Displays the menu screen if the game has stopeed propting the user to either restart or go to the home menu
    */
    void displayDeathMenu();
    /** Determines the action to be taken depending on the button input once the game has ended
    */
    void deathButtonSelections();
    /** Determines the action to be taken depending on the button input at the home screen
    */
    void homeButtonSelections();
    /** Disables the A button for 300 milliseconds after being pressed to prevent erroneous input from switch bouncing
    *@param a boolean indicating whether the button has been pressed
    */
    void disableAButton(bool a);
    /** Disables the Y button for 300 milliseconds after being pressed to prevent erroneous input from switch bouncing
    *@param a boolean indicating whether the button has been pressed
    */
    void disableYButton(bool y);
    /** Disables the B button for 300 milliseconds after being pressed to prevent erroneous input from switch bouncing
    *@param a boolean indicating whether the button has been pressed
    */
    void disableBButton(bool b);
    /** Is called using a ticker 300 milliseconds after the A button is pressed to re-enable it
    */
    void enableA();
    /** Is called using a ticker 300 milliseconds after the Y button is pressed to re-enable it
    */
    void enableY();
    /** Is called using a ticker 300 milliseconds after the B button is pressed to re-enable it
    */
    void enableB();
    /** Process the joystick and Y, B and A button inputs and store their values in a structure
    */
    void processInput();
    /** Begin the execution of the game
    */
    void play();
    /** Displays the home screen
    */
    void homeScreen();
    /** Display the help screens
    */
    void helpScreen();
    /** Check whether the user has advanced to the next help screen by pressing the A button
    */
    void checkNextHelpScreen();

  public:
    /** The constructor of the Game class and is used to execute the game.
    */
   Game();
    /** Executes the main loop
    */
   void run();


};