#ifndef SPRITES_H
#define SPRITES_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/////////// All sprite bitmaps ////////////////

// right facing miner bitmap
const int miner_right[24] =   {
        
        1,1,1,
        1,1,0,
        1,1,1,
        0,1,0,
        1,1,1,
        1,1,1,
        0,1,0,
        0,1,1,

    };
// left facing miner bitmap
const int miner_left[24] =   {
        
        1,1,1,
        0,1,1,
        1,1,1,
        0,1,0,
        1,1,1,
        1,1,1,
        0,1,0,
        1,1,0,
    };
// enemy bitmap
const int enemy[15] =   {
        
        1,1,1,
        1,0,1,
        1,1,1,
        0,1,0,
        1,1,1,
    };
// key bitmap
const int key[12] =   {
        
        1,1,0,0,
        1,0,1,1,
        1,1,0,1,
    };
// blank bitmap to show key collected
const int key_collected[12] =   {
        
        0,0,0,0,
        0,0,0,0,
        0,0,0,0,
    };
// trap bitmap
const int spike[9] =   {
        
        1,0,1,
        0,1,0,
        1,0,1,
    };
// solid block bitmap
const int solid_block[18] =   {
        
        1,1,1,1,1,1,
        1,0,1,1,0,1,
        1,1,1,1,1,1,
    };
// level exit bitmap
const int door[30] = {
        
        1,1,1,1,1,
        1,0,1,0,1,
        1,1,0,1,1,
        1,0,1,0,1,
        1,1,0,1,1,
        1,0,1,0,1,
};


///////////////////////////////////////////////////////////////////////////////

/** Sprites Class
@details Draws sprites and sets all collision rules 
@author Andrew Milner University of Leeds
@date April 2019
*/ 

class Sprites
{

public:
    /** Constructor
    */
    Sprites();
    /** Deconstructor
    */ 
    ~Sprites();
    /** Gets x and y position of player.
    */
    Vector2D get_pos();
    /** States starting position of player.
    @param x miner position.
    @param y miner position.
    */
    void miner_init(int x, int y);
    /** Updates player x position.
    @details Updates x positon depending on direction of joystick. Will only update 
    * position if no collision detected. Function also stops player leaving screen 
    * at the left and right sides of screen.
    @param d Direction enum from Gamepad library.
    */
    void miner_move(Direction d, N5110 &lcd);
    /** Draws sprite facing direction moving.
    @details _direction states if player left or right facing, function draws
    * left or right facing sprite according.
    */
    void miner_draw(N5110 &lcd);
    /** States when player is on a platform.
    @details if _jump true then player can jump, this prevents mid-air jumping.
    * Also states conditions for gravity so player will fall unless on top
    * of a platform.
    */    
    void miner_land(N5110 &lcd);
    /** Contains jump conditions and updates player position while jumping.
    @details When A button pressed players y position increases until preset value
    * is reached. Jump flag is true while player y is increasing and false all other
    * times, this prevents double jumping.
    */
    void miner_jump(N5110 &lcd, Gamepad &pad);
    /** Player falls if not on platform.
    */
    void miner_gravity(N5110 &lcd);
    /** initialises enemies
    @param i index these will always be 0, 1, 2 so each enemy is treated individually.
    @param x position.
    @param y position.
    @param distance in pixels enemy to travel.
    */
    void enemy_init(int i, int x, int y, int d);
    /** Updates enemy movement.
    @param i index these will always be 0 - 2 so each enemy is treated individually.
    @param velocity how many pixels enemies will move each loop.
    */
    void enemy_move(int i, double v, N5110 &lcd);
    /** States conditions for collision between enemy and player.
    @details uses get_pos() and detects if any overlap between player and enemies.
    @param i index these will always be 0, 1, 2 so each enemy is treated individually.
    @return will return true if collision detected.
    */
    bool enemy_collision(int i);
    /** States conditions for drawing and collection of keys.
    @details Each key is displayed while key flag is false, once collected, key is deleted 
    * and flag is changed to true.
    @param k index so each key is treated independently
    @param x key position.
    @param y key position.
    */
    void key_collect(int k, int x, int y, N5110 &lcd, Gamepad &pad);
    /** Counts keys collected.
    @return number of keys collected.
    */
    int keys_collected();
    /** makes _keys zero
    @details When game over occurs keys need to be reset to zero
    */
    void zero_keys();
    /** Draws level exit and detects player collision
    @details takes player position from get_pos() and detects if overlap between
    player and the exit sprite.
    @param x exit position.
    @param y exit position.
    */
    bool exit_level(int x, int y, N5110 &lcd);
    /** Draws traps and detects player collision
    @details takes player position from get_pos() and detects if overlap between
    player and the trap sprite.
    @param x trap position.
    @param y trap position.
    */
    bool trap(int x, int y, N5110 &lcd);
    /** Draws blocks and detects player collision.
    @param d direction so player can turn around if collision with block.
    @Param i index so collision rules between blocks dont clash, will always be numbered 0 - 4.
    @param x block position.
    @param y block position.
    */
    void blocks(Direction d, int i, int x, int y, N5110 &lcd);
    /** Checks if player has collided with any blocks.
    @return if player is contact with any block returns true.
    */
    bool block_collision();
    /** Draws sinking blocks and detects player contact.
    @param x1 sinking block starting position.
    @param y sinking block position.
    @param x2 Sinking block finish position.
    */
    void soft_blocks(int x1, int y, int x2, N5110 &lcd);
    
    bool _key[5]; /*initialise key flag, will only draw key if false, array identifies keys 
    individually, made global variable as key_reinit() needs to reset flag at start of each level*/
    
private:

    int _direction;
    bool _gravity;
    bool _jump;
    int _y;
    int _x;
    bool _j_flag;
    int _j_counter;
    
    double _kx[5]; // key x position
    double _ky[5]; // key y position
    int _keys; // keeps count of how many keys collected
    
    bool _eflag[5];  // initialise enemy flag, array identifies each enemy individually 
    double _ex[5];   // enemy x position 
    double _ey[5];   // enemy y position 
    int _counter[5]; // enemy counter, counts pixels moved 
    int _distance[5];// distance enemy will travel before turning round 
    
    double _bx[5];   // block x position array identifies each block individually 
    double _by[5];   // block y position array 
    bool _collision[5]; // collision indicator for each individual block 
       
};
#endif