#ifndef GAMEENGINE_H
#define GAMEENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Sprites.h"
#include "Levels.h"

/** Gameengine Class
@details Builds all levels and sets game default values. Updates all game settings 
and contains functions for game conditions.

@author Andrew Milner University of Leeds

@date April 2019
*/ 
class Gameengine
{

public:
    /** constructor
    */
    Gameengine();
    /** deconstructor
    */
    ~Gameengine();
    /** Initialises all default game settings.
    * @details Sets player starting position, number of lives, level start and timer.
    */
    void game_init();
    /** Updates all game game settings.
    * @details Contains all functions that update player position and game conditions
    such as lives left, timer, game over and level complete.
    */
    void update(N5110 &lcd, Gamepad &pad);
    /** Updates and renders all level objects for level 1.
    */
    void draw_l1(N5110 &lcd, Gamepad &pad);
    /** Updates and renders all level objects for level 2.
    */
    void draw_l2(N5110 &lcd, Gamepad &pad);
    /** Updates and renders all level objects for level 3.
    */
    void draw_l3(N5110 &lcd, Gamepad &pad);
    /** Updates and renders all level objects for level 4.
    */
    void draw_l4(N5110 &lcd, Gamepad &pad);
    /** 
    * @brief Reads direction the player is moving.
    * @details Uses get_direction() function from Gamepad library.
    */
    void read_direction(Gamepad &pad);
    /** Calculates lives remaining.
    * @returns integer value to control lives fsm state in leds() main.cpp
    */
    int lives_leds();
    /** Calculates time remaining.
    * @details Calculates time remaining, every 1/3 of total time passed the return 
    * value changes.
    * @return integer value to control oxygen fsm state in leds() main.cpp
    */
    int oxygen_leds();
    /** 
    * @brief Calculates and displays player score.
    * @details At game over or game completed the players score is calculated and
    * displayed. It takes time remaining and multiplies this by lives remaining, then 
    * adds on 10 points for every key collected.
    */ 
    void get_score(N5110 &lcd);
    /** Returns true if game completed.
    * @details calculates when _level variable has gone beyond number of levels in game.
    * @return true if game complete
    */ 
    bool game_complete(N5110 &lcd);
    /** Reduces number of lives on player death.
    * @details After reducing lives, makes tone to signify player death, resets player
    * back to start, creates small pause.
    */
    void lose_life(Gamepad &pad, N5110 &lcd);
    /** States whether game over condition met
    * @return returns true if game over.
    */
    bool game_over();
    /** Returns true when player at exit.
    * @return Function returns true when the player is in contact with the level exit.
    */
    bool level_exit(N5110 &lcd);
    /** Moves player to next level when level complete conditions met.
    * @details When all 5 keys have been collected using _sprites.keys_collected()
    * and player is in contact with the level exit using level_exit() function.
    * Calculates time reamining (for score) then resets, resets enemy and key flags
    * so they can be initialised for the next level, increases _level by 1 so next
    * level will be drawn, keys needed increased by 5 and miner position reset.
    */
    void next_level(N5110 &lcd, Gamepad &pad);
    /** Returns true if player in contact with trap
    * @details Function sets all trap poisitons and collision rules. Usues int i
    * to select which level positions to use.
    */
    bool trap_death(N5110 &lcd);
    /** Draws keys for each level.
    * @details Function sets key positions and collision rules stated in
    * Sprites class. Uses int i to select which level positions to use, these are
    * specified in Level class.
    */
    void key_draw(N5110 &lcd,Gamepad &pad);
    /** Sets all block positions and collision rules.
    * @details Function sets all block positons and collision rules stated in
    * Sprites class. Blocks are indexed so each one is treated independently
    * otherwise all will disappear if one is collected. Uses int i to select which
    * level positions to use, these are specified in Level class. Int i is set _level
    * so function knows which level settings to use.
    */
    void blocks(N5110 &lcd);
    /** Initialises enemies, updates movements and detects collision with player.
    * @details Only initialises enemies when enem_flag set to false and sets it true
    * once done. Function also updates enemy position and detects collision with players,
    * int i states which level settings to use.
    * @return true if player collides with enemy.
    */
    bool enemies(N5110 &lcd);
    /** Reinitialises keys once new level starts.
    * @details Once level is complete the _key_reint flag is set to false and all
    * 5 level keys are set to false. This means they are drawn in new positions as
    * per int i which states which level settings to use in key_draw().  
    */
    void key_reinit();

private:
    
    int _level;
    int _lives;
    bool _enem_flag;
    int _five_keys;
    bool _key_reinit;
    int _oxy_state;
    int _life_state;
    float _total_time;
    float _time;
    
    Sprites _sprites;
    Direction _d;
    Levels _lev;
    Timer _t;
    
    
};
#endif