#ifndef UI_H
#define UI_H

namespace Menus {
    /**
     * UI class
     * @brief Provides a user-friendly abstraction layer above the graphics
     * namespace, allowing the programmer to add labels, buttons and other 
     * UI elements to the 'canvas', without worrying about positioning or 
     * input management.
     * @see Graphics::UI
     */
    class UI {
    public:
        /**
         * @brief Creates a new UI object ready to be used
         */
        UI();
        
        /**
         * @brief Destructor
         */
        ~UI();
        
        /**
         * @brief Should be called before a new Menu is loaded, to reset state
         * such as the cursor position.
         */
        void reset();
        
        /**
         * @brief Should be called every frame, prepares the 'canvas' for a new
         * set of 'draw' calls.
         */
        void clear();
        
        /**
         * @brief Draws a 'title' style label to the screen, essentially the
         * same as drawLabel.
         * @param text The text of the title, must be <= 8 characters
         * @see drawLabel
         */
        void drawTitle(const char * text);
        
        /**
         * @brief Draws a label to the canvas and advance the 'canvas pointer'.
         * @param text The text of the label, must be <= 8 characters
         */
        void drawLabel(const char * text);
        
        /**
         * @brief 
         * @param text The text of the flashing label, must be <= 8 characters
         * @see drawLabel
         */
        void drawFlashingLabel(const char * text);
        
        /**
         * @brief
         * @param text The text of the button, must be <= 8 characters
         * @returns true if the button is pressed this frame, else false.
         */
        bool drawAndCheckButton(const char * text);
        
        /**
         * @brief Draws the big inverted 'TETRIS' logo and advances the canvas
         * pointer.
         */
        void drawLogo();
    
        /**
         * @brief Called to move the 'cursor' (highlighted button) down to
         * the next one vertically beneath.
         */
        void selectNextButton();
        
        /**
         * @brief Called to move the 'cursor' (highlighted button) up to
         * the next one vertically above.
         */
        void selectPreviousButton();
        
        /**
         * @brief "Presses" the current button, so it's draw function
         * will return true.
         */
        void pressButton();
        
        /**
         * @brief Simply adds vertical whitespace to the canvas.
         */
        void newLine();
    
    private:
        /**
         * The amount of X padding between the top of the screen and UI elements
         */
        static const int PADDING_X = 3;
        
        /**
         * The padding between the edge of the screen and UI elements
         */
        static const int PADDING_Y = 4;
        
        /**
         * The padding between buttons
         */
        static const int LINE_PADDING = 3;
        
        /**
         * Intefaced with nextY()
         */
        int next_y_pos;
        
        /**
         * @brief Increments next_y_pos
         * @returns The next y position to put a button at
         */
        int nextY();
        
        /**
         * The button currently selected in the list.
         */
        int button_currently_selected;
        
        /**
         * The number being drawn currently, starting at 0, resets to zero 
         * every frame.
         */
        int button_currently_drawing;
        
        /**
         * True if the 'OK button' has been clicked this frame
         */
        bool button_pressed_frame;
        
        /**
         * The current frame counter, used for flashing text
         */
        int frame;
    };
};
#endif