#include "mbed.h"

/**
 * SampledIn class
 * @brief Essentially wraps the DigitalIn class, providing an abstraction
 * layer for a "De-Bounced" input reader. This is achived by sampling the 
 * pin to check for stable signals. The tradeoff for debouncing the pin, is a
 * slower response time.
 */
class SampledIn {
public:
    /**
     * @brief Create a new SampledIn object
     * @param pin The pin to use as input, just like with a DigitalIn object
     * @see DigitalIn
     */
    SampledIn(PinName pin);
    
    /**
     * Destructor
     * @brief deletes the DigitalIn object this wraps
     */
    ~SampledIn();
        
    /**
     * @brief Check the status of the pin
     * @returns True if the button at this input's pin was hit
     * in this sampling window
     */
    bool hit_frame();
    
private:
    /**
     * The frequency to sample the DigitalIn at.
     */
    static const double freq = 0.005;
    
    /**
     * The minimum number of frames the input has to be stable at for it to
     * be registered as valid.
     */
    static const int min_frames = 8;
    
    /**
     * Pointer to input button reader.
     */
    DigitalIn * in;
    
    /**
     * The last valid output of the digital in.
     */
    int last_in;
    
    /**
     * The current stable output returned.
     */
    int stable_output;
    
    /**
     * Called to look at the value of the digital input.
     */
    void sample();
    
    /**
     * The number of frames the output has been stable for.
     */
    int stable_frames;
    
    /**
     * The ticker used to call sample every 1/freq seconds
     * @see freq
     */
    Ticker ticker;
    
    /**
     * The last value that hit_frame() returned..
     */
    bool last_value_returned;
};